//*****************************************************************************
//
//! @file ios_lram_host.c
//!
//! @brief Example host used for demonstrating the use of the IOS LRAM.
//!
//! Purpose: This host component runs on one EVB and is used in conjunction with
//! the companion slave example, ios_lram, which runs on a second EVB.
//!
//! The host example uses the ITM SWO to let the user know progress and
//! status of the demonstration.  The SWO is configured at 1M baud.
//!
//! Printing takes place over the ITM at 1M Baud.
//!
//! Additional Information:
//! In order to run this example, a slave device (e.g. a second EVB) must be set
//! up to run the companion example, ios_lram.  The two boards can be connected
//! using fly leads between the two boards as follows.
//!
//! @verbatim
//! Pin connections for the I/O Master board to the I/O Slave board.
//! SPI:
//!     HOST (ios_lram_host)                    SLAVE (ios_lram)
//!     --------------------                    ----------------
//!     GPIO[10] GPIO Interrupt (slave to host) GPIO[4]  GPIO interrupt
//!     GPIO[5]  IOM0 SPI SCK                   GPIO[0]  IOS SPI SCK
//!     GPIO[7]  IOM0 SPI MOSI                  GPIO[1]  IOS SPI MOSI
//!     GPIO[6]  IOM0 SPI MISO                  GPIO[2]  IOS SPI MISO
//!     GPIO[11] IOM0 SPI nCE                   GPIO[3]  IOS SPI nCE
//!     GND                                     GND
//!
//! I2C:
//!     HOST (ios_lram_host)                    SLAVE (ios_lram)
//!     --------------------                    ----------------
//!     GPIO[10] GPIO Interrupt (slave to host) GPIO[4]  GPIO interrupt
//!     GPIO[5]  IOM0 I2C SCL                   GPIO[0]  IOS I2C SCL
//!     GPIO[6]  IOM0 I2C SDA                   GPIO[1]  IOS I2C SDA
//!     GND                                     GND
//! @endverbatim
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2019, Ambiq Micro
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
// 
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision v2.2.0-7-g63f7c2ba1 of the AmbiqSuite Development Package.
//
//*****************************************************************************
#include <string.h>
#include <stdint.h>
#include <stdbool.h>
#include "am_mcu_apollo.h"
#include "am_bsp.h"
#include "am_util.h"

#define IOM_MODULE                  0
#define USE_SPI                     1   // 0 = I2C, 1 = SPI
#define I2C_ADDR                    0x10

#define AM_IOSTEST_CMD_START_DATA   0xFD
#define AM_IOSTEST_CMD_ACK_DATA     0xFE
#define AM_IOSTEST_CMD_STOP_DATA    0xFF

#define AM_IOSTEST_WRITE_CMD_BIT    0x80

#define IOSOFFSET_INTEN             0x78
#define IOSOFFSET_INTSTAT           0x79
#define IOSOFFSET_INTCLR            0x7A
#define IOSOFFSET_INTSET            0x7B
#define IOSOFFSET_HANDSHAKE         0

#define HANDSHAKE_IOS_TO_IOM        2
#define INBUFFER_EMPTY              (0xEE)
#define TEST_XOR_BYTE               0xFF // 0 Will not change, 0xFF will invert

#define HANDSHAKE_PIN               10
#define IOM_TRIGGER_GPIO            0

#define TEST_ARRAY_SIZE             7640

const uint8_t g_pui8TestArray[TEST_ARRAY_SIZE] =
{
    0x00, 0x02, 0x00, 0x10, 0xC9, 0x94, 0x00, 0x00,
    0x35, 0x95, 0x00, 0x00, 0xD5, 0x81, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x05, 0x81, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x3D, 0x95, 0x00, 0x00, 0x3D, 0x95, 0x00, 0x00,
    0x80, 0xB5, 0x82, 0xB0, 0x00, 0xAF, 0x02, 0x20,
    0x01, 0xF0, 0xC8, 0xFA, 0x00, 0x20, 0x4F, 0xF6,
    0xFF, 0x71, 0x01, 0xF0, 0x49, 0xFB, 0x00, 0x20,
    0x09, 0x49, 0x01, 0xF0, 0xCF, 0xFA, 0x08, 0x23,
    0x7B, 0x60, 0x7B, 0x68, 0x5B, 0x08, 0x00, 0x20,
    0x4F, 0xF6, 0xFF, 0x71, 0x7A, 0x68, 0x01, 0xF0,
    0x45, 0xFB, 0x01, 0x20, 0x01, 0xF0, 0x8A, 0xFB,
    0x08, 0x37, 0xBD, 0x46, 0x80, 0xBD, 0x00, 0xBF,
    0x00, 0x02, 0x00, 0x10, 0x80, 0xB5, 0x00, 0xAF,
    0x01, 0x20, 0x01, 0xF0, 0x7F, 0xFB, 0x05, 0x4B,
    0x1B, 0x68, 0x01, 0x33, 0x03, 0x4A, 0x13, 0x60,
    0x0F, 0x2B, 0x02, 0xD9, 0x01, 0x4B, 0x00, 0x22,
    0x1A, 0x60, 0x80, 0xBD, 0x7C, 0x02, 0x00, 0x10,
    0x80, 0xB5, 0x00, 0xAF, 0x00, 0x20, 0x01, 0xF0,
    0x6D, 0xFA, 0x01, 0xF0, 0x07, 0xFA, 0x20, 0x48,
    0x00, 0xF0, 0xC2, 0xF8, 0x1F, 0x4B, 0x73, 0x22,
    0x1A, 0x60, 0x1F, 0x4B, 0x1E, 0x4A, 0x12, 0x68,
    0x22, 0xF0, 0x70, 0x02, 0x1A, 0x60, 0x1D, 0x4B,
    0x1C, 0x4A, 0x12, 0x68, 0x22, 0xF4, 0x7F, 0x42,
    0x42, 0xF4, 0x80, 0x52, 0x1A, 0x60, 0x17, 0x4B,
    0x00, 0x22, 0x1A, 0x60, 0x18, 0x48, 0x04, 0x21,
    0x01, 0xF0, 0x8E, 0xF8, 0x01, 0xF0, 0xA2, 0xFB,
    0x01, 0xF0, 0xFA, 0xF9, 0x00, 0xF0, 0x3A, 0xFF,
    0x14, 0x48, 0x00, 0xF0, 0x19, 0xFF, 0xFF, 0xF7,
    0x9F, 0xFF, 0x01, 0x20, 0x01, 0xF0, 0x3A, 0xFB,
    0x1D, 0x20, 0x01, 0xF0, 0x4F, 0xFB, 0x01, 0xF0,
    0x75, 0xFB, 0x00, 0x20, 0x4F, 0xF6, 0xFF, 0x71,
    0x01, 0xF0, 0xD4, 0xFA, 0x01, 0xF0, 0x0C, 0xFA,
    0x01, 0x20, 0x01, 0xF0, 0xCD, 0xFC, 0x0A, 0x4B,
    0x1B, 0x68, 0x07, 0x48, 0x04, 0x21, 0x1A, 0x46,
    0x01, 0xF0, 0x64, 0xF9, 0xF4, 0xE7, 0x00, 0xBF,
    0x09, 0x96, 0x00, 0x00, 0x60, 0x00, 0x01, 0x40,
    0x54, 0x00, 0x01, 0x40, 0x28, 0x00, 0x01, 0x40,
    0x0C, 0x02, 0x00, 0x10, 0xCC, 0x9C, 0x00, 0x00,
    0x7C, 0x02, 0x00, 0x10, 0x80, 0xB5, 0x68, 0x46,
    0x08, 0x30, 0x00, 0xF0, 0x09, 0xF8, 0x01, 0xBD,
    0x18, 0x46, 0x00, 0xBF, 0x80, 0x00, 0x40, 0x18,
    0x00, 0x68, 0x70, 0x47, 0x18, 0x46, 0x00, 0xBF,
    0x80, 0xB5, 0x96, 0xB0, 0x00, 0xAF, 0x78, 0x60,
    0x00, 0x23, 0x7B, 0x65, 0x7B, 0x6D, 0xFB, 0x64,
    0x2D, 0x4B, 0x1B, 0x68, 0xFB, 0x64, 0xFB, 0x6C,
    0xDB, 0xB2, 0x87, 0xF8, 0x50, 0x30, 0xFB, 0x6C,
    0x1B, 0x0A, 0xDB, 0xB2, 0x87, 0xF8, 0x51, 0x30,
    0xFB, 0x6C, 0x1B, 0x0C, 0x9B, 0xB2, 0xA7, 0xF8,
    0x52, 0x30, 0x26, 0x4B, 0x1B, 0x68, 0xBB, 0x64,
    0x97, 0xF8, 0x51, 0x30, 0xDB, 0xB2, 0x03, 0xF0,
    0x02, 0x03, 0x00, 0x2B, 0x05, 0xD0, 0x06, 0x20,
    0x79, 0x68, 0xFF, 0xF7, 0xD3, 0xFF, 0x03, 0x46,
    0x01, 0xE0, 0x4F, 0xF0, 0xFF, 0x33, 0x7B, 0x64,
    0x00, 0x20, 0x79, 0x68, 0xFF, 0xF7, 0xCA, 0xFF,
    0x03, 0x46, 0x7B, 0x62, 0x01, 0x20, 0x79, 0x68,
    0xFF, 0xF7, 0xC4, 0xFF, 0x03, 0x46, 0xBB, 0x62,
    0x02, 0x20, 0x79, 0x68, 0xFF, 0xF7, 0xBE, 0xFF,
    0x03, 0x46, 0xFB, 0x62, 0x03, 0x20, 0x79, 0x68,
    0xFF, 0xF7, 0xB8, 0xFF, 0x03, 0x46, 0x3B, 0x63,
    0x04, 0x20, 0x79, 0x68, 0xFF, 0xF7, 0xB2, 0xFF,
    0x03, 0x46, 0x7B, 0x63, 0x05, 0x20, 0x79, 0x68,
    0xFF, 0xF7, 0xAC, 0xFF, 0x03, 0x46, 0xBB, 0x63,
    0x06, 0x20, 0x79, 0x68, 0xFF, 0xF7, 0xA6, 0xFF,
    0x03, 0x46, 0xFB, 0x63, 0x07, 0x20, 0x79, 0x68,
    0xFF, 0xF7, 0xA0, 0xFF, 0x03, 0x46, 0x3B, 0x64,
    0x07, 0xF1, 0x0C, 0x03, 0x18, 0x46, 0x01, 0xF0,
    0x5B, 0xFB, 0x00, 0x23, 0x7B, 0x65, 0xFE, 0xE7,
    0x28, 0xED, 0x00, 0xE0, 0x38, 0xED, 0x00, 0xE0,
    0x80, 0xB4, 0x83, 0xB0, 0x00, 0xAF, 0x78, 0x60,
    0x03, 0x4B, 0x7A, 0x68, 0x1A, 0x60, 0x0C, 0x37,
    0xBD, 0x46, 0x5D, 0xF8, 0x04, 0x7B, 0x70, 0x47,
    0x88, 0x03, 0x00, 0x10, 0x2D, 0xE9, 0xB0, 0x0F,
    0x97, 0xB0, 0x00, 0xAF, 0xC7, 0xE9, 0x0C, 0x01,
    0xD7, 0xE9, 0x0C, 0x23, 0x1C, 0x00, 0x00, 0x25,
    0x54, 0xEA, 0x05, 0x00, 0x00, 0xF0, 0x94, 0x80,
    0xD7, 0xE9, 0x0C, 0x23, 0x59, 0x08, 0x4F, 0xEA,
    0x32, 0x00, 0xD7, 0xE9, 0x0C, 0x23, 0x4F, 0xEA,
    0x92, 0x08, 0x48, 0xEA, 0x83, 0x78, 0x4F, 0xEA,
    0x93, 0x09, 0x10, 0xEB, 0x08, 0x02, 0x41, 0xEB,
    0x09, 0x03, 0xC7, 0xE9, 0x14, 0x23, 0xD7, 0xE9,
    0x14, 0x23, 0x4F, 0xEA, 0x12, 0x1A, 0x4A, 0xEA,
    0x03, 0x7A, 0x4F, 0xEA, 0x13, 0x1B, 0xD7, 0xE9,
    0x14, 0x23, 0x12, 0xEB, 0x0A, 0x02, 0x43, 0xEB,
    0x0B, 0x03, 0xC7, 0xE9, 0x14, 0x23, 0xD7, 0xE9,
    0x14, 0x23, 0x11, 0x0A, 0xB9, 0x62, 0xB8, 0x6A,
    0x40, 0xEA, 0x03, 0x60, 0xB8, 0x62, 0x19, 0x0A,
    0xF9, 0x62, 0xD7, 0xE9, 0x14, 0x23, 0xD7, 0xE9,
    0x0A, 0x01, 0x12, 0x18, 0x43, 0xEB, 0x01, 0x03,
    0xC7, 0xE9, 0x14, 0x23, 0xD7, 0xE9, 0x14, 0x23,
    0x11, 0x0C, 0x39, 0x62, 0x38, 0x6A, 0x40, 0xEA,
    0x03, 0x40, 0x38, 0x62, 0x19, 0x0C, 0x79, 0x62,
    0xD7, 0xE9, 0x14, 0x23, 0xD7, 0xE9, 0x08, 0x01,
    0x12, 0x18, 0x43, 0xEB, 0x01, 0x03, 0xC7, 0xE9,
    0x14, 0x23, 0xD7, 0xE9, 0x14, 0x23, 0x19, 0x00,
    0xB9, 0x61, 0x00, 0x22, 0xFA, 0x61, 0xD7, 0xE9,
    0x14, 0x23, 0xD7, 0xE9, 0x06, 0x01, 0x12, 0x18,
    0x43, 0xEB, 0x01, 0x03, 0xC7, 0xE9, 0x14, 0x23,
    0xD7, 0xE9, 0x14, 0x23, 0xD1, 0x08, 0xB9, 0x60,
    0xB8, 0x68, 0x40, 0xEA, 0x43, 0x70, 0xB8, 0x60,
    0xD9, 0x08, 0xF9, 0x60, 0xD7, 0xE9, 0x02, 0x23,
    0xC7, 0xE9, 0x14, 0x23, 0xD7, 0xE9, 0x14, 0x23,
    0x92, 0x18, 0x43, 0xEB, 0x03, 0x03, 0x98, 0x00,
    0x78, 0x60, 0x79, 0x68, 0x41, 0xEA, 0x92, 0x71,
    0x79, 0x60, 0x90, 0x00, 0x38, 0x60, 0xD7, 0xE9,
    0x00, 0x01, 0x12, 0x18, 0x43, 0xEB, 0x01, 0x03,
    0xD7, 0xE9, 0x0C, 0x01, 0x82, 0x1A, 0x61, 0xEB,
    0x03, 0x03, 0xC7, 0xE9, 0x12, 0x23, 0xD7, 0xE9,
    0x12, 0x23, 0x06, 0x32, 0x43, 0xF1, 0x00, 0x03,
    0x11, 0x09, 0x39, 0x61, 0x38, 0x69, 0x40, 0xEA,
    0x03, 0x70, 0x38, 0x61, 0x19, 0x09, 0x79, 0x61,
    0xD7, 0xE9, 0x14, 0x23, 0xD7, 0xE9, 0x04, 0x01,
    0x12, 0x18, 0x43, 0xEB, 0x01, 0x03, 0x29, 0xE0,
    0x3B, 0x6B, 0x7B, 0x64, 0x7B, 0x6C, 0x5A, 0x08,
    0x7B, 0x6C, 0x9B, 0x08, 0x13, 0x44, 0x3B, 0x64,
    0x3B, 0x6C, 0x1B, 0x09, 0x3A, 0x6C, 0x13, 0x44,
    0x3B, 0x64, 0x3B, 0x6C, 0x1B, 0x0A, 0x3A, 0x6C,
    0x13, 0x44, 0x3B, 0x64, 0x3B, 0x6C, 0x1B, 0x0C,
    0x3A, 0x6C, 0x13, 0x44, 0x3B, 0x64, 0x3B, 0x6C,
    0xDB, 0x08, 0x3B, 0x64, 0x3A, 0x6C, 0x13, 0x46,
    0x9B, 0x00, 0x13, 0x44, 0x5B, 0x00, 0x7A, 0x6C,
    0xD3, 0x1A, 0xFB, 0x63, 0xFB, 0x6B, 0x06, 0x33,
    0x1A, 0x09, 0x3B, 0x6C, 0x13, 0x44, 0x1A, 0x46,
    0x4F, 0xF0, 0x00, 0x03, 0x10, 0x46, 0x19, 0x46,
    0x5C, 0x37, 0xBD, 0x46, 0xBD, 0xE8, 0xB0, 0x0F,
    0x70, 0x47, 0x00, 0xBF, 0x80, 0xB5, 0x84, 0xB0,
    0x00, 0xAF, 0xC7, 0xE9, 0x00, 0x01, 0x3A, 0x68,
    0x7B, 0x68, 0x13, 0x43, 0x00, 0x2B, 0x14, 0xBF,
    0x00, 0x23, 0x01, 0x23, 0xDB, 0xB2, 0xFB, 0x60,
    0x08, 0xE0, 0xD7, 0xE9, 0x00, 0x01, 0xFF, 0xF7,
    0x19, 0xFF, 0xC7, 0xE9, 0x00, 0x01, 0xFB, 0x68,
    0x01, 0x33, 0xFB, 0x60, 0xD7, 0xE9, 0x00, 0x23,
    0x52, 0xEA, 0x03, 0x01, 0xF1, 0xD1, 0xFB, 0x68,
    0x18, 0x46, 0x10, 0x37, 0xBD, 0x46, 0x80, 0xBD,
    0x80, 0xB5, 0x82, 0xB0, 0x00, 0xAF, 0xC7, 0xE9,
    0x00, 0x01, 0xD7, 0xE9, 0x00, 0x23, 0x00, 0x2A,
    0x73, 0xF1, 0x00, 0x01, 0x06, 0xDA, 0xD7, 0xE9,
    0x00, 0x23, 0x52, 0x42, 0x63, 0xEB, 0x43, 0x03,
    0xC7, 0xE9, 0x00, 0x23, 0xD7, 0xE9, 0x00, 0x23,
    0x10, 0x46, 0x19, 0x46, 0xFF, 0xF7, 0xC6, 0xFF,
    0x03, 0x46, 0x18, 0x46, 0x08, 0x37, 0xBD, 0x46,
    0x80, 0xBD, 0x00, 0xBF, 0x80, 0xB4, 0x85, 0xB0,
    0x00, 0xAF, 0xC7, 0xE9, 0x00, 0x01, 0x38, 0x68,
    0x79, 0x68, 0x01, 0x43, 0x00, 0x29, 0x14, 0xBF,
    0x00, 0x21, 0x01, 0x21, 0xC9, 0xB2, 0xF9, 0x60,
    0x0A, 0xE0, 0xD7, 0xE9, 0x00, 0x01, 0x02, 0x09,
    0x42, 0xEA, 0x01, 0x72, 0x0B, 0x09, 0xC7, 0xE9,
    0x00, 0x23, 0xF9, 0x68, 0x01, 0x31, 0xF9, 0x60,
    0xD7, 0xE9, 0x00, 0x01, 0x50, 0xEA, 0x01, 0x0C,
    0xEF, 0xD1, 0xFB, 0x68, 0x18, 0x46, 0x14, 0x37,
    0xBD, 0x46, 0x5D, 0xF8, 0x04, 0x7B, 0x70, 0x47,
    0x80, 0xB4, 0x87, 0xB0, 0x00, 0xAF, 0x78, 0x60,
    0x39, 0x60, 0x00, 0x23, 0xFB, 0x75, 0x00, 0x23,
    0x3B, 0x61, 0x00, 0x23, 0xFB, 0x60, 0x7B, 0x68,
    0x1B, 0x78, 0x2D, 0x2B, 0x08, 0xD1, 0x01, 0x23,
    0xFB, 0x75, 0x7B, 0x68, 0x01, 0x33, 0x7B, 0x60,
    0xFB, 0x68, 0x01, 0x33, 0xFB, 0x60, 0x13, 0xE0,
    0x12, 0xE0, 0xFB, 0x68, 0x01, 0x33, 0xFB, 0x60,
    0x3A, 0x69, 0x13, 0x46, 0x9B, 0x00, 0x13, 0x44,
    0x5B, 0x00, 0x3B, 0x61, 0x7B, 0x68, 0x1B, 0x78,
    0x1A, 0x46, 0x3B, 0x69, 0x13, 0x44, 0x30, 0x3B,
    0x3B, 0x61, 0x7B, 0x68, 0x01, 0x33, 0x7B, 0x60,
    0x7B, 0x68, 0x1B, 0x78, 0x2F, 0x2B, 0x03, 0xD9,
    0x7B, 0x68, 0x1B, 0x78, 0x39, 0x2B, 0xE4, 0xD9,
    0x3B, 0x68, 0x00, 0x2B, 0x02, 0xD0, 0x3B, 0x68,
    0xFA, 0x68, 0x1A, 0x60, 0xFB, 0x7D, 0x00, 0x2B,
    0x02, 0xD0, 0x3B, 0x69, 0x5B, 0x42, 0x00, 0xE0,
    0x3B, 0x69, 0x18, 0x46, 0x1C, 0x37, 0xBD, 0x46,
    0x5D, 0xF8, 0x04, 0x7B, 0x70, 0x47, 0x00, 0xBF,
    0x80, 0xB5, 0x90, 0xB0, 0x00, 0xAF, 0xC7, 0xE9,
    0x02, 0x01, 0x7A, 0x60, 0x00, 0x23, 0xFB, 0x63,
    0x00, 0x23, 0xBB, 0x63, 0xD7, 0xE9, 0x02, 0x01,
    0xFF, 0xF7, 0x70, 0xFE, 0xC7, 0xE9, 0x0C, 0x01,
    0xB9, 0x68, 0x3A, 0x6B, 0x13, 0x46, 0x9B, 0x00,
    0x13, 0x44, 0x5B, 0x00, 0xCB, 0x1A, 0xFB, 0x62,
    0xFB, 0x6B, 0x5A, 0x1C, 0xFA, 0x63, 0xFA, 0x6A,
    0xD2, 0xB2, 0x30, 0x32, 0xD2, 0xB2, 0x07, 0xF1,
    0x40, 0x01, 0x0B, 0x44, 0x03, 0xF8, 0x30, 0x2C,
    0xD7, 0xE9, 0x0C, 0x23, 0xC7, 0xE9, 0x02, 0x23,
    0xD7, 0xE9, 0x02, 0x23, 0x52, 0xEA, 0x03, 0x01,
    0xDC, 0xD1, 0xFB, 0x6B, 0xBB, 0x63, 0x7B, 0x68,
    0x00, 0x2B, 0x11, 0xD0, 0x08, 0xE0, 0x7B, 0x68,
    0x5A, 0x1C, 0x7A, 0x60, 0x07, 0xF1, 0x10, 0x01,
    0xFA, 0x6B, 0x0A, 0x44, 0x12, 0x78, 0x1A, 0x70,
    0xFB, 0x6B, 0x5A, 0x1E, 0xFA, 0x63, 0x00, 0x2B,
    0xF1, 0xD1, 0x7B, 0x68, 0x00, 0x22, 0x1A, 0x70,
    0xBB, 0x6B, 0x18, 0x46, 0x40, 0x37, 0xBD, 0x46,
    0x80, 0xBD, 0x00, 0xBF, 0xB0, 0xB4, 0x8D, 0xB0,
    0x00, 0xAF, 0xC7, 0xE9, 0x02, 0x01, 0x7A, 0x60,
    0xFB, 0x70, 0x00, 0x23, 0xFB, 0x62, 0xD7, 0xE9,
    0x02, 0x23, 0x52, 0xEA, 0x03, 0x01, 0x09, 0xD1,
    0xFB, 0x6A, 0x5A, 0x1C, 0xFA, 0x62, 0x07, 0xF1,
    0x30, 0x02, 0x13, 0x44, 0x30, 0x22, 0x03, 0xF8,
    0x20, 0x2C, 0x28, 0xE0, 0x27, 0xE0, 0x3B, 0x7A,
    0x03, 0xF0, 0x0F, 0x03, 0x87, 0xF8, 0x2B, 0x30,
    0x97, 0xF8, 0x2B, 0x30, 0x09, 0x2B, 0x0A, 0xD9,
    0xFB, 0x78, 0x00, 0x2B, 0x01, 0xD0, 0x27, 0x23,
    0x00, 0xE0, 0x07, 0x23, 0x97, 0xF8, 0x2B, 0x20,
    0x13, 0x44, 0x87, 0xF8, 0x2B, 0x30, 0xFB, 0x6A,
    0x5A, 0x1C, 0xFA, 0x62, 0x97, 0xF8, 0x2B, 0x20,
    0x30, 0x32, 0xD2, 0xB2, 0x07, 0xF1, 0x30, 0x01,
    0x0B, 0x44, 0x03, 0xF8, 0x20, 0x2C, 0xD7, 0xE9,
    0x02, 0x23, 0x14, 0x09, 0x44, 0xEA, 0x03, 0x74,
    0x1D, 0x09, 0xC7, 0xE9, 0x02, 0x45, 0xD7, 0xE9,
    0x02, 0x23, 0x52, 0xEA, 0x03, 0x01, 0xD2, 0xD1,
    0xFB, 0x6A, 0x7B, 0x62, 0x7B, 0x68, 0x00, 0x2B,
    0x11, 0xD0, 0x08, 0xE0, 0x7B, 0x68, 0x5A, 0x1C,
    0x7A, 0x60, 0x07, 0xF1, 0x10, 0x01, 0xFA, 0x6A,
    0x0A, 0x44, 0x12, 0x78, 0x1A, 0x70, 0xFB, 0x6A,
    0x5A, 0x1E, 0xFA, 0x62, 0x00, 0x2B, 0xF1, 0xD1,
    0x7B, 0x68, 0x00, 0x22, 0x1A, 0x70, 0x7B, 0x6A,
    0x18, 0x46, 0x34, 0x37, 0xBD, 0x46, 0xB0, 0xBC,
    0x70, 0x47, 0x00, 0xBF, 0x80, 0xB4, 0x85, 0xB0,
    0x00, 0xAF, 0x78, 0x60, 0x00, 0x23, 0xFB, 0x60,
    0x7B, 0x68, 0x00, 0x2B, 0x01, 0xD1, 0xFB, 0x68,
    0x0A, 0xE0, 0x02, 0xE0, 0xFB, 0x68, 0x01, 0x33,
    0xFB, 0x60, 0x7B, 0x68, 0x5A, 0x1C, 0x7A, 0x60,
    0x1B, 0x78, 0x00, 0x2B, 0xF6, 0xD1, 0xFB, 0x68,
    0x18, 0x46, 0x14, 0x37, 0xBD, 0x46, 0x5D, 0xF8,
    0x04, 0x7B, 0x70, 0x47, 0x80, 0xB4, 0x87, 0xB0,
    0x00, 0xAF, 0xF8, 0x60, 0x0B, 0x46, 0x7A, 0x60,
    0xFB, 0x72, 0x00, 0x23, 0x7B, 0x61, 0x7B, 0x68,
    0x00, 0x2B, 0x01, 0xDC, 0x7B, 0x69, 0x11, 0xE0,
    0x0A, 0xE0, 0xFB, 0x68, 0x00, 0x2B, 0x04, 0xD0,
    0xFB, 0x68, 0x5A, 0x1C, 0xFA, 0x60, 0xFA, 0x7A,
    0x1A, 0x70, 0x7B, 0x69, 0x01, 0x33, 0x7B, 0x61,
    0x7B, 0x68, 0x5A, 0x1E, 0x7A, 0x60, 0x00, 0x2B,
    0xEF, 0xD1, 0x7B, 0x69, 0x18, 0x46, 0x1C, 0x37,
    0xBD, 0x46, 0x5D, 0xF8, 0x04, 0x7B, 0x70, 0x47,
    0x80, 0xB5, 0x8E, 0xB0, 0x00, 0xAF, 0xF8, 0x60,
    0xB9, 0x60, 0x7A, 0x60, 0xBB, 0x68, 0x1B, 0x68,
    0xBB, 0x62, 0xBB, 0x6A, 0x03, 0x2B, 0x02, 0xDC,
    0x6F, 0xF0, 0x02, 0x03, 0xD6, 0xE0, 0xD7, 0xED,
    0x03, 0x7A, 0xF5, 0xEE, 0x40, 0x7A, 0xF1, 0xEE,
    0x10, 0xFA, 0x04, 0xD1, 0xBB, 0x68, 0x69, 0x4A,
    0x1A, 0x60, 0x03, 0x23, 0xCA, 0xE0, 0xBB, 0x68,
    0x7B, 0x62, 0xFB, 0x68, 0x7B, 0x61, 0x7B, 0x69,
    0xDB, 0x15, 0xDB, 0xB2, 0x7F, 0x3B, 0x3B, 0x62,
    0x7B, 0x69, 0xC3, 0xF3, 0x16, 0x03, 0x43, 0xF4,
    0x00, 0x03, 0xFB, 0x61, 0x00, 0x23, 0x3B, 0x63,
    0x00, 0x23, 0x7B, 0x63, 0x3B, 0x6A, 0x1E, 0x2B,
    0x02, 0xDD, 0x6F, 0xF0, 0x01, 0x03, 0xB1, 0xE0,
    0x3B, 0x6A, 0x13, 0xF1, 0x17, 0x0F, 0x02, 0xDA,
    0x4F, 0xF0, 0xFF, 0x33, 0xAA, 0xE0, 0x3B, 0x6A,
    0x16, 0x2B, 0x06, 0xDD, 0x3B, 0x6A, 0x17, 0x3B,
    0xFA, 0x69, 0x02, 0xFA, 0x03, 0xF3, 0x7B, 0x63,
    0x1A, 0xE0, 0x3B, 0x6A, 0x00, 0x2B, 0x0F, 0xDB,
    0x3B, 0x6A, 0xC3, 0xF1, 0x17, 0x03, 0xFA, 0x69,
    0x42, 0xFA, 0x03, 0xF3, 0x7B, 0x63, 0x3B, 0x6A,
    0x01, 0x33, 0xFA, 0x69, 0x02, 0xFA, 0x03, 0xF3,
    0x23, 0xF0, 0x7F, 0x43, 0x3B, 0x63, 0x07, 0xE0,
    0xFB, 0x69, 0x23, 0xF0, 0x7F, 0x42, 0x3B, 0x6A,
    0xDB, 0x43, 0x42, 0xFA, 0x03, 0xF3, 0x3B, 0x63,
    0x7B, 0x69, 0x00, 0x2B, 0x04, 0xDA, 0xBB, 0x68,
    0x5A, 0x1C, 0xBA, 0x60, 0x2D, 0x22, 0x1A, 0x70,
    0x7B, 0x6B, 0x00, 0x2B, 0x05, 0xD1, 0xBB, 0x68,
    0x5A, 0x1C, 0xBA, 0x60, 0x30, 0x22, 0x1A, 0x70,
    0x23, 0xE0, 0x7B, 0x6B, 0x00, 0x2B, 0x09, 0xDD,
    0x7B, 0x6B, 0x1A, 0x46, 0x4F, 0xEA, 0xE2, 0x73,
    0x10, 0x46, 0x19, 0x46, 0xBA, 0x68, 0xFF, 0xF7,
    0x93, 0xFE, 0x12, 0xE0, 0xBB, 0x68, 0x5A, 0x1C,
    0xBA, 0x60, 0x2D, 0x22, 0x1A, 0x70, 0x7B, 0x6B,
    0x5B, 0x42, 0x1A, 0x46, 0x4F, 0xEA, 0xE2, 0x73,
    0x10, 0x46, 0x19, 0x46, 0xBA, 0x68, 0xFF, 0xF7,
    0x83, 0xFE, 0x02, 0xE0, 0xBB, 0x68, 0x01, 0x33,
    0xBB, 0x60, 0xBB, 0x68, 0x1B, 0x78, 0x00, 0x2B,
    0xF8, 0xD1, 0xBB, 0x68, 0x5A, 0x1C, 0xBA, 0x60,
    0x2E, 0x22, 0x1A, 0x70, 0x3B, 0x6B, 0x00, 0x2B,
    0x05, 0xD1, 0xBB, 0x68, 0x5A, 0x1C, 0xBA, 0x60,
    0x30, 0x22, 0x1A, 0x70, 0x3C, 0xE0, 0x7A, 0x6A,
    0xBB, 0x68, 0xD2, 0x1A, 0xBB, 0x6A, 0x13, 0x44,
    0x01, 0x3B, 0xBB, 0x61, 0xBA, 0x69, 0x7B, 0x68,
    0x9A, 0x42, 0xB8, 0xBF, 0x13, 0x46, 0xBB, 0x61,
    0x00, 0x23, 0xFB, 0x62, 0x15, 0xE0, 0x3A, 0x6B,
    0x13, 0x46, 0x9B, 0x00, 0x13, 0x44, 0x5B, 0x00,
    0x3B, 0x63, 0xBB, 0x68, 0x5A, 0x1C, 0xBA, 0x60,
    0x3A, 0x6B, 0x12, 0x16, 0xD2, 0xB2, 0x30, 0x32,
    0xD2, 0xB2, 0x1A, 0x70, 0x3B, 0x6B, 0x23, 0xF0,
    0x7F, 0x43, 0x3B, 0x63, 0xFB, 0x6A, 0x01, 0x33,
    0xFB, 0x62, 0xFA, 0x6A, 0xBB, 0x69, 0x9A, 0x42,
    0xE5, 0xDB, 0xBB, 0x68, 0x01, 0x3B, 0xBB, 0x60,
    0x02, 0xE0, 0xBB, 0x68, 0x01, 0x3B, 0xBB, 0x60,
    0xBB, 0x68, 0x1B, 0x78, 0x30, 0x2B, 0x04, 0xD1,
    0xBB, 0x68, 0x01, 0x3B, 0x1B, 0x78, 0x2E, 0x2B,
    0xF3, 0xD1, 0xBB, 0x68, 0x01, 0x33, 0xBB, 0x60,
    0xBB, 0x68, 0x00, 0x22, 0x1A, 0x70, 0xBA, 0x68,
    0x7B, 0x6A, 0xD3, 0x1A, 0x18, 0x46, 0x38, 0x37,
    0xBD, 0x46, 0x80, 0xBD, 0x30, 0x2E, 0x30, 0x00,
    0x80, 0xB5, 0x94, 0xB0, 0x00, 0xAF, 0xF8, 0x60,
    0xB9, 0x60, 0x7A, 0x60, 0x00, 0x23, 0xFB, 0x63,
    0x00, 0x23, 0x7B, 0x62, 0xE9, 0xE2, 0x06, 0x23,
    0x3B, 0x63, 0xBB, 0x68, 0x1B, 0x78, 0x25, 0x2B,
    0x1F, 0xD0, 0xFB, 0x68, 0x00, 0x2B, 0x15, 0xD0,
    0xBB, 0x68, 0x1B, 0x78, 0x0A, 0x2B, 0x0B, 0xD1,
    0xB2, 0x4B, 0x1B, 0x78, 0x00, 0x2B, 0x07, 0xD0,
    0xFB, 0x68, 0x5A, 0x1C, 0xFA, 0x60, 0x0D, 0x22,
    0x1A, 0x70, 0xFB, 0x6B, 0x01, 0x33, 0xFB, 0x63,
    0xFB, 0x68, 0x5A, 0x1C, 0xFA, 0x60, 0xBA, 0x68,
    0x12, 0x78, 0x1A, 0x70, 0xBB, 0x68, 0x01, 0x33,
    0xBB, 0x60, 0xFB, 0x6B, 0x01, 0x33, 0xFB, 0x63,
    0xC3, 0xE2, 0xBB, 0x68, 0x01, 0x33, 0xBB, 0x60,
    0x00, 0x23, 0x87, 0xF8, 0x2D, 0x30, 0x97, 0xF8,
    0x2D, 0x30, 0x87, 0xF8, 0x2E, 0x30, 0x20, 0x23,
    0x87, 0xF8, 0x2F, 0x30, 0xBB, 0x68, 0x1B, 0x78,
    0x30, 0x2B, 0x05, 0xD1, 0x30, 0x23, 0x87, 0xF8,
    0x2F, 0x30, 0xBB, 0x68, 0x01, 0x33, 0xBB, 0x60,
    0x07, 0xF1, 0x10, 0x03, 0xB8, 0x68, 0x19, 0x46,
    0xFF, 0xF7, 0x8E, 0xFD, 0x03, 0x46, 0xBB, 0x63,
    0x3B, 0x69, 0xBA, 0x68, 0x13, 0x44, 0xBB, 0x60,
    0xBB, 0x68, 0x1B, 0x78, 0x73, 0x2B, 0x05, 0xD0,
    0xBB, 0x6B, 0x00, 0x2B, 0x02, 0xDA, 0xBB, 0x6B,
    0x5B, 0x42, 0xBB, 0x63, 0xBB, 0x68, 0x1B, 0x78,
    0x2E, 0x2B, 0x0E, 0xD1, 0xBB, 0x68, 0x01, 0x33,
    0xBB, 0x60, 0x07, 0xF1, 0x10, 0x03, 0xB8, 0x68,
    0x19, 0x46, 0xFF, 0xF7, 0x71, 0xFD, 0x03, 0x46,
    0x3B, 0x63, 0x3B, 0x69, 0xBA, 0x68, 0x13, 0x44,
    0xBB, 0x60, 0xBB, 0x68, 0x1B, 0x78, 0x6C, 0x2B,
    0x0C, 0xD1, 0xBB, 0x68, 0x01, 0x33, 0xBB, 0x60,
    0xBB, 0x68, 0x1B, 0x78, 0x6C, 0x2B, 0x05, 0xD1,
    0xBB, 0x68, 0x01, 0x33, 0xBB, 0x60, 0x01, 0x23,
    0x87, 0xF8, 0x2D, 0x30, 0xBB, 0x68, 0x1B, 0x78,
    0x46, 0x3B, 0x32, 0x2B, 0x00, 0xF2, 0x5D, 0x82,
    0x01, 0xA1, 0x51, 0xF8, 0x23, 0xF0, 0x00, 0xBF,
    0xE9, 0x8E, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x77, 0x8C, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x7D, 0x8B, 0x00, 0x00,
    0xAD, 0x8D, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0xE9, 0x8E, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0xAD, 0x8D, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0xA3, 0x8B, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x17, 0x8D, 0x00, 0x00,
    0x63, 0x8F, 0x00, 0x00, 0x63, 0x8F, 0x00, 0x00,
    0x71, 0x8C, 0x00, 0x00, 0x7B, 0x68, 0x1A, 0x1D,
    0x7A, 0x60, 0x1B, 0x68, 0x87, 0xF8, 0x23, 0x30,
    0xFB, 0x68, 0x00, 0x2B, 0x05, 0xD0, 0xFB, 0x68,
    0x5A, 0x1C, 0xFA, 0x60, 0x97, 0xF8, 0x23, 0x20,
    0x1A, 0x70, 0xFB, 0x6B, 0x01, 0x33, 0xFB, 0x63,
    0xEC, 0xE1, 0x7B, 0x68, 0x1A, 0x1D, 0x7A, 0x60,
    0x1B, 0x68, 0xFB, 0x64, 0xF8, 0x6C, 0xFF, 0xF7,
    0xC5, 0xFD, 0x78, 0x62, 0xBB, 0x6B, 0x00, 0x2B,
    0x1F, 0xDD, 0xBA, 0x6B, 0x7B, 0x6A, 0x9A, 0x42,
    0x1B, 0xD9, 0xBA, 0x6B, 0x7B, 0x6A, 0xD3, 0x1A,
    0xBB, 0x63, 0x97, 0xF8, 0x2F, 0x30, 0xF8, 0x68,
    0x19, 0x46, 0xBA, 0x6B, 0xFF, 0xF7, 0xCE, 0xFD,
    0xB8, 0x63, 0xFB, 0x68, 0x00, 0x2B, 0x01, 0xD0,
    0xBB, 0x6B, 0x00, 0xE0, 0x00, 0x23, 0xFA, 0x68,
    0x13, 0x44, 0xFB, 0x60, 0xBB, 0x6B, 0xFA, 0x6B,
    0x13, 0x44, 0xFB, 0x63, 0x00, 0x23, 0xBB, 0x63,
    0x0F, 0xE0, 0x0E, 0xE0, 0xFB, 0x68, 0x00, 0x2B,
    0x05, 0xD0, 0xFB, 0x68, 0x5A, 0x1C, 0xFA, 0x60,
    0xFA, 0x6C, 0x12, 0x78, 0x1A, 0x70, 0xFB, 0x6C,
    0x01, 0x33, 0xFB, 0x64, 0xFB, 0x6B, 0x01, 0x33,
    0xFB, 0x63, 0xFB, 0x6C, 0x1B, 0x78, 0x00, 0x2B,
    0xEC, 0xD1, 0xBB, 0x6B, 0x00, 0x2B, 0x22, 0xD0,
    0xBB, 0x6B, 0x5B, 0x42, 0xBB, 0x63, 0xBA, 0x6B,
    0x7B, 0x6A, 0x9A, 0x42, 0x1B, 0xD9, 0xBA, 0x6B,
    0x7B, 0x6A, 0xD3, 0x1A, 0xBB, 0x63, 0x97, 0xF8,
    0x2F, 0x30, 0xF8, 0x68, 0x19, 0x46, 0xBA, 0x6B,
    0xFF, 0xF7, 0x94, 0xFD, 0xB8, 0x63, 0xFB, 0x68,
    0x00, 0x2B, 0x01, 0xD0, 0xBB, 0x6B, 0x00, 0xE0,
    0x00, 0x23, 0xFA, 0x68, 0x13, 0x44, 0xFB, 0x60,
    0xBB, 0x6B, 0xFA, 0x6B, 0x13, 0x44, 0xFB, 0x63,
    0x00, 0x23, 0xBB, 0x63, 0x86, 0xE1, 0x85, 0xE1,
    0x01, 0x23, 0x87, 0xF8, 0x2E, 0x30, 0x97, 0xF8,
    0x2D, 0x30, 0x00, 0x2B, 0x0C, 0xD0, 0x7B, 0x68,
    0x07, 0x33, 0x23, 0xF0, 0x07, 0x03, 0x03, 0xF1,
    0x08, 0x02, 0x7A, 0x60, 0xD3, 0xE9, 0x00, 0x23,
    0x09, 0xE0, 0x00, 0xBF, 0x80, 0x03, 0x00, 0x10,
    0x7B, 0x68, 0x1A, 0x1D, 0x7A, 0x60, 0x1B, 0x68,
    0x1A, 0x46, 0x4F, 0xF0, 0x00, 0x03, 0xC7, 0xE9,
    0x10, 0x23, 0xBB, 0x6B, 0x00, 0x2B, 0x1E, 0xD0,
    0xD7, 0xE9, 0x10, 0x01, 0xFF, 0xF7, 0x26, 0xFC,
    0x03, 0x46, 0xBA, 0x6B, 0xD3, 0x1A, 0xBB, 0x63,
    0x97, 0xF8, 0x2F, 0x30, 0xF8, 0x68, 0x19, 0x46,
    0xBA, 0x6B, 0xFF, 0xF7, 0x53, 0xFD, 0xB8, 0x63,
    0xFB, 0x68, 0x00, 0x2B, 0x01, 0xD0, 0xBB, 0x6B,
    0x00, 0xE0, 0x00, 0x23, 0xFA, 0x68, 0x13, 0x44,
    0xFB, 0x60, 0xBB, 0x6B, 0xFA, 0x6B, 0x13, 0x44,
    0xFB, 0x63, 0x00, 0x23, 0xBB, 0x63, 0x97, 0xF8,
    0x2E, 0x30, 0xD7, 0xE9, 0x10, 0x01, 0xFA, 0x68,
    0xFF, 0xF7, 0xBC, 0xFC, 0x78, 0x63, 0xFB, 0x68,
    0x00, 0x2B, 0x03, 0xD0, 0x7B, 0x6B, 0xFA, 0x68,
    0x13, 0x44, 0xFB, 0x60, 0x7B, 0x6B, 0xFA, 0x6B,
    0x13, 0x44, 0xFB, 0x63, 0x32, 0xE1, 0x97, 0xF8,
    0x2D, 0x30, 0x00, 0x2B, 0x09, 0xD0, 0x7B, 0x68,
    0x07, 0x33, 0x23, 0xF0, 0x07, 0x03, 0x03, 0xF1,
    0x08, 0x02, 0x7A, 0x60, 0xD3, 0xE9, 0x00, 0x23,
    0x06, 0xE0, 0x7B, 0x68, 0x1A, 0x1D, 0x7A, 0x60,
    0x1B, 0x68, 0x1A, 0x46, 0x4F, 0xF0, 0x00, 0x03,
    0xC7, 0xE9, 0x10, 0x23, 0xBB, 0x6B, 0x00, 0x2B,
    0x1E, 0xD0, 0xD7, 0xE9, 0x10, 0x01, 0xFF, 0xF7,
    0x99, 0xFB, 0x03, 0x46, 0xBA, 0x6B, 0xD3, 0x1A,
    0xBB, 0x63, 0x97, 0xF8, 0x2F, 0x30, 0xF8, 0x68,
    0x19, 0x46, 0xBA, 0x6B, 0xFF, 0xF7, 0x06, 0xFD,
    0xB8, 0x63, 0xFB, 0x68, 0x00, 0x2B, 0x01, 0xD0,
    0xBB, 0x6B, 0x00, 0xE0, 0x00, 0x23, 0xFA, 0x68,
    0x13, 0x44, 0xFB, 0x60, 0xBB, 0x6B, 0xFA, 0x6B,
    0x13, 0x44, 0xFB, 0x63, 0x00, 0x23, 0xBB, 0x63,
    0xD7, 0xE9, 0x10, 0x01, 0xFA, 0x68, 0xFF, 0xF7,
    0x27, 0xFC, 0x78, 0x63, 0xFB, 0x68, 0x00, 0x2B,
    0x03, 0xD0, 0x7B, 0x6B, 0xFA, 0x68, 0x13, 0x44,
    0xFB, 0x60, 0x7B, 0x6B, 0xFA, 0x6B, 0x13, 0x44,
    0xFB, 0x63, 0xE7, 0xE0, 0x97, 0xF8, 0x2D, 0x30,
    0x00, 0x2B, 0x09, 0xD0, 0x7B, 0x68, 0x07, 0x33,
    0x23, 0xF0, 0x07, 0x03, 0x03, 0xF1, 0x08, 0x02,
    0x7A, 0x60, 0xD3, 0xE9, 0x00, 0x23, 0x06, 0xE0,
    0x7B, 0x68, 0x1A, 0x1D, 0x7A, 0x60, 0x1B, 0x68,
    0x1A, 0x46, 0x4F, 0xEA, 0xE2, 0x73, 0xC7, 0xE9,
    0x06, 0x23, 0xD7, 0xE9, 0x06, 0x23, 0x00, 0x2A,
    0x73, 0xF1, 0x00, 0x01, 0x0A, 0xDA, 0xD7, 0xE9,
    0x06, 0x23, 0x52, 0x42, 0x63, 0xEB, 0x43, 0x03,
    0xC7, 0xE9, 0x10, 0x23, 0x01, 0x23, 0x87, 0xF8,
    0x2C, 0x30, 0x06, 0xE0, 0xD7, 0xE9, 0x06, 0x23,
    0xC7, 0xE9, 0x10, 0x23, 0x00, 0x23, 0x87, 0xF8,
    0x2C, 0x30, 0xBB, 0x6B, 0x00, 0x2B, 0x4A, 0xD0,
    0xD7, 0xE9, 0x10, 0x23, 0x10, 0x46, 0x19, 0x46,
    0xFF, 0xF7, 0x56, 0xFB, 0x03, 0x46, 0xBA, 0x6B,
    0xD3, 0x1A, 0xBB, 0x63, 0x97, 0xF8, 0x2C, 0x30,
    0x00, 0x2B, 0x11, 0xD0, 0xBB, 0x6B, 0x01, 0x3B,
    0xBB, 0x63, 0x97, 0xF8, 0x2F, 0x30, 0x30, 0x2B,
    0x0A, 0xD1, 0xFB, 0x68, 0x00, 0x2B, 0x04, 0xD0,
    0xFB, 0x68, 0x5A, 0x1C, 0xFA, 0x60, 0x2D, 0x22,
    0x1A, 0x70, 0xFB, 0x6B, 0x01, 0x33, 0xFB, 0x63,
    0x97, 0xF8, 0x2F, 0x30, 0xF8, 0x68, 0x19, 0x46,
    0xBA, 0x6B, 0xFF, 0xF7, 0x8B, 0xFC, 0xB8, 0x63,
    0xFB, 0x68, 0x00, 0x2B, 0x01, 0xD0, 0xBB, 0x6B,
    0x00, 0xE0, 0x00, 0x23, 0xFA, 0x68, 0x13, 0x44,
    0xFB, 0x60, 0xBB, 0x6B, 0xFA, 0x6B, 0x13, 0x44,
    0xFB, 0x63, 0x00, 0x23, 0xBB, 0x63, 0x97, 0xF8,
    0x2C, 0x30, 0x00, 0x2B, 0x1E, 0xD0, 0x97, 0xF8,
    0x2F, 0x30, 0x20, 0x2B, 0x1A, 0xD1, 0xFB, 0x68,
    0x00, 0x2B, 0x04, 0xD0, 0xFB, 0x68, 0x5A, 0x1C,
    0xFA, 0x60, 0x2D, 0x22, 0x1A, 0x70, 0xFB, 0x6B,
    0x01, 0x33, 0xFB, 0x63, 0x0E, 0xE0, 0x97, 0xF8,
    0x2C, 0x30, 0x00, 0x2B, 0x0A, 0xD0, 0xFB, 0x68,
    0x00, 0x2B, 0x04, 0xD0, 0xFB, 0x68, 0x5A, 0x1C,
    0xFA, 0x60, 0x2D, 0x22, 0x1A, 0x70, 0xFB, 0x6B,
    0x01, 0x33, 0xFB, 0x63, 0xD7, 0xE9, 0x10, 0x01,
    0xFA, 0x68, 0xFF, 0xF7, 0x89, 0xFB, 0x78, 0x63,
    0xFB, 0x68, 0x00, 0x2B, 0x03, 0xD0, 0x7B, 0x6B,
    0xFA, 0x68, 0x13, 0x44, 0xFB, 0x60, 0x7B, 0x6B,
    0xFA, 0x6B, 0x13, 0x44, 0xFB, 0x63, 0x49, 0xE0,
    0xFB, 0x68, 0x00, 0x2B, 0x38, 0xD0, 0x7B, 0x68,
    0x07, 0x33, 0x23, 0xF0, 0x07, 0x03, 0x03, 0xF1,
    0x08, 0x02, 0x7A, 0x60, 0xD3, 0xE9, 0x00, 0x23,
    0x10, 0x46, 0x19, 0x46, 0x00, 0xF0, 0x92, 0xFE,
    0x03, 0x46, 0x7B, 0x61, 0xFB, 0x68, 0x14, 0x22,
    0x1A, 0x60, 0x78, 0x69, 0xF9, 0x68, 0x3A, 0x6B,
    0xFF, 0xF7, 0x52, 0xFC, 0x78, 0x63, 0x7B, 0x6B,
    0x00, 0x2B, 0x14, 0xDA, 0x7B, 0x6B, 0xB3, 0xF1,
    0xFF, 0x3F, 0x02, 0xD1, 0x1D, 0x4B, 0xBB, 0x62,
    0x08, 0xE0, 0x7B, 0x6B, 0x13, 0xF1, 0x02, 0x0F,
    0x02, 0xD1, 0x1B, 0x4B, 0xBB, 0x62, 0x01, 0xE0,
    0x1A, 0x4B, 0xBB, 0x62, 0xFB, 0x68, 0xBA, 0x6A,
    0x1A, 0x60, 0x03, 0x23, 0x7B, 0x63, 0x7B, 0x6B,
    0xFA, 0x6B, 0x13, 0x44, 0xFB, 0x63, 0x7B, 0x6B,
    0xFA, 0x68, 0x13, 0x44, 0xFB, 0x60, 0x0D, 0xE0,
    0x0C, 0xE0, 0xFB, 0x68, 0x00, 0x2B, 0x05, 0xD0,
    0xFB, 0x68, 0x5A, 0x1C, 0xFA, 0x60, 0xBA, 0x68,
    0x12, 0x78, 0x1A, 0x70, 0xFB, 0x6B, 0x01, 0x33,
    0xFB, 0x63, 0x00, 0xBF, 0xBB, 0x68, 0x01, 0x33,
    0xBB, 0x60, 0xBB, 0x68, 0x1B, 0x78, 0x00, 0x2B,
    0x7F, 0xF4, 0x11, 0xAD, 0xFB, 0x68, 0x00, 0x2B,
    0x02, 0xD0, 0xFB, 0x68, 0x00, 0x22, 0x1A, 0x70,
    0xFB, 0x6B, 0x18, 0x46, 0x50, 0x37, 0xBD, 0x46,
    0x80, 0xBD, 0x00, 0xBF, 0x30, 0x2E, 0x30, 0x00,
    0x23, 0x2E, 0x23, 0x00, 0x3F, 0x2E, 0x3F, 0x00,
    0x0F, 0xB4, 0x80, 0xB5, 0x82, 0xB0, 0x00, 0xAF,
    0x07, 0xF1, 0x14, 0x03, 0x3B, 0x60, 0x09, 0x48,
    0x39, 0x69, 0x3A, 0x68, 0xFF, 0xF7, 0xE8, 0xFC,
    0x78, 0x60, 0x07, 0x4B, 0x1B, 0x68, 0x05, 0x48,
    0x98, 0x47, 0x7B, 0x68, 0x18, 0x46, 0x08, 0x37,
    0xBD, 0x46, 0xBD, 0xE8, 0x80, 0x40, 0x04, 0xB0,
    0x70, 0x47, 0x00, 0xBF, 0x80, 0x02, 0x00, 0x10,
    0x88, 0x03, 0x00, 0x10, 0x80, 0xB5, 0x00, 0xAF,
    0x03, 0x48, 0xFF, 0xF7, 0xDD, 0xFF, 0x03, 0x48,
    0xFF, 0xF7, 0xDA, 0xFF, 0x80, 0xBD, 0x00, 0xBF,
    0xE4, 0x9C, 0x00, 0x00, 0x04, 0x9D, 0x00, 0x00,
    0x90, 0xB4, 0x82, 0xB0, 0x00, 0xAF, 0x78, 0x60,
    0x7B, 0x68, 0x5B, 0x68, 0x03, 0xF0, 0x02, 0x03,
    0x00, 0x2B, 0x00, 0xF0, 0x94, 0x80, 0x7B, 0x68,
    0x1B, 0x68, 0x00, 0x2B, 0x00, 0xDA, 0x4D, 0xE0,
    0x92, 0x4B, 0x73, 0x22, 0x1A, 0x60, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0xF8, 0x03, 0x5A, 0x08,
    0x8F, 0x4B, 0x13, 0x44, 0x1A, 0x46, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x07, 0x03, 0x9B, 0x00,
    0x02, 0x21, 0x99, 0x40, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0xF8, 0x03, 0x58, 0x08, 0x88, 0x4B,
    0x03, 0x44, 0x18, 0x68, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0x07, 0x03, 0x9B, 0x00, 0x07, 0x24,
    0x04, 0xFA, 0x03, 0xF3, 0xDB, 0x43, 0x03, 0x40,
    0x0B, 0x43, 0x13, 0x60, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0xFC, 0x03, 0x03, 0xF1, 0x80, 0x43,
    0x03, 0xF5, 0x80, 0x33, 0x1A, 0x46, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x03, 0x03, 0xDB, 0x00,
    0x18, 0x21, 0x99, 0x40, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0xFC, 0x03, 0x03, 0xF1, 0x80, 0x43,
    0x03, 0xF5, 0x80, 0x33, 0x18, 0x68, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x03, 0x03, 0xDB, 0x00,
    0xFF, 0x24, 0x04, 0xFA, 0x03, 0xF3, 0xDB, 0x43,
    0x03, 0x40, 0x0B, 0x43, 0x13, 0x60, 0x6D, 0x4B,
    0x00, 0x22, 0x1A, 0x60, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0x20, 0x03, 0xDA, 0x08, 0x6B, 0x4B,
    0x13, 0x44, 0x7A, 0x68, 0x12, 0x68, 0x02, 0xF0,
    0x1F, 0x02, 0x01, 0x21, 0x01, 0xFA, 0x02, 0xF2,
    0x1A, 0x60, 0x7B, 0x68, 0x1B, 0x68, 0x03, 0xF0,
    0x20, 0x03, 0xDA, 0x08, 0x64, 0x4B, 0x13, 0x44,
    0x1A, 0x46, 0x7B, 0x68, 0x5B, 0x68, 0x03, 0xF0,
    0x01, 0x03, 0x00, 0x2B, 0x10, 0xD0, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xD9, 0x08,
    0x5D, 0x4B, 0x0B, 0x44, 0x19, 0x68, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x1F, 0x03, 0x01, 0x20,
    0x00, 0xFA, 0x03, 0xF3, 0x0B, 0x43, 0x10, 0xE0,
    0x7B, 0x68, 0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03,
    0xD9, 0x08, 0x55, 0x4B, 0x0B, 0x44, 0x19, 0x68,
    0x7B, 0x68, 0x1B, 0x68, 0x03, 0xF0, 0x1F, 0x03,
    0x01, 0x20, 0x00, 0xFA, 0x03, 0xF3, 0xDB, 0x43,
    0x0B, 0x40, 0x13, 0x60, 0x92, 0xE0, 0x7B, 0x68,
    0x1B, 0x68, 0x00, 0x2B, 0x00, 0xDA, 0x4D, 0xE0,
    0x48, 0x4B, 0x73, 0x22, 0x1A, 0x60, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0xF8, 0x03, 0x5A, 0x08,
    0x45, 0x4B, 0x13, 0x44, 0x1A, 0x46, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x07, 0x03, 0x9B, 0x00,
    0x06, 0x21, 0x99, 0x40, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0xF8, 0x03, 0x58, 0x08, 0x3E, 0x4B,
    0x03, 0x44, 0x18, 0x68, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0x07, 0x03, 0x9B, 0x00, 0x07, 0x24,
    0x04, 0xFA, 0x03, 0xF3, 0xDB, 0x43, 0x03, 0x40,
    0x0B, 0x43, 0x13, 0x60, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0xFC, 0x03, 0x03, 0xF1, 0x80, 0x43,
    0x03, 0xF5, 0x80, 0x33, 0x1A, 0x46, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x03, 0x03, 0xDB, 0x00,
    0x18, 0x21, 0x99, 0x40, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0xFC, 0x03, 0x03, 0xF1, 0x80, 0x43,
    0x03, 0xF5, 0x80, 0x33, 0x18, 0x68, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x03, 0x03, 0xDB, 0x00,
    0xFF, 0x24, 0x04, 0xFA, 0x03, 0xF3, 0xDB, 0x43,
    0x03, 0x40, 0x0B, 0x43, 0x13, 0x60, 0x23, 0x4B,
    0x00, 0x22, 0x1A, 0x60, 0x7B, 0x68, 0x1B, 0x68,
    0x03, 0xF0, 0x20, 0x03, 0xDA, 0x08, 0x23, 0x4B,
    0x13, 0x44, 0x7A, 0x68, 0x12, 0x68, 0x02, 0xF0,
    0x1F, 0x02, 0x01, 0x21, 0x01, 0xFA, 0x02, 0xF2,
    0x1A, 0x60, 0x7B, 0x68, 0x1B, 0x68, 0x03, 0xF0,
    0x20, 0x03, 0xDA, 0x08, 0x1A, 0x4B, 0x13, 0x44,
    0x1A, 0x46, 0x7B, 0x68, 0x5B, 0x68, 0x03, 0xF0,
    0x01, 0x03, 0x00, 0x2B, 0x10, 0xD0, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xD9, 0x08,
    0x13, 0x4B, 0x0B, 0x44, 0x19, 0x68, 0x7B, 0x68,
    0x1B, 0x68, 0x03, 0xF0, 0x1F, 0x03, 0x01, 0x20,
    0x00, 0xFA, 0x03, 0xF3, 0x0B, 0x43, 0x10, 0xE0,
    0x7B, 0x68, 0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03,
    0xD9, 0x08, 0x0B, 0x4B, 0x0B, 0x44, 0x19, 0x68,
    0x7B, 0x68, 0x1B, 0x68, 0x03, 0xF0, 0x1F, 0x03,
    0x01, 0x20, 0x00, 0xFA, 0x03, 0xF3, 0xDB, 0x43,
    0x0B, 0x40, 0x13, 0x60, 0x08, 0x37, 0xBD, 0x46,
    0x90, 0xBC, 0x70, 0x47, 0x60, 0x00, 0x01, 0x40,
    0x40, 0x00, 0x01, 0x40, 0xA8, 0x00, 0x01, 0x40,
    0x88, 0x00, 0x01, 0x40, 0xB4, 0x00, 0x01, 0x40,
    0x80, 0xB5, 0x84, 0xB0, 0x00, 0xAF, 0x78, 0x60,
    0x39, 0x60, 0x00, 0x23, 0xFB, 0x60, 0x09, 0xE0,
    0xFB, 0x68, 0xDB, 0x00, 0x7A, 0x68, 0x13, 0x44,
    0x18, 0x46, 0xFF, 0xF7, 0xB1, 0xFE, 0xFB, 0x68,
    0x01, 0x33, 0xFB, 0x60, 0xFA, 0x68, 0x3B, 0x68,
    0x9A, 0x42, 0xF1, 0xD3, 0x10, 0x37, 0xBD, 0x46,
    0x80, 0xBD, 0x00, 0xBF, 0x80, 0xB4, 0x83, 0xB0,
    0x00, 0xAF, 0x78, 0x60, 0x39, 0x60, 0x3B, 0x68,
    0xDB, 0x00, 0x7A, 0x68, 0x13, 0x44, 0x5B, 0x68,
    0x03, 0xF0, 0x02, 0x03, 0x00, 0x2B, 0x43, 0xD0,
    0x3B, 0x68, 0xDB, 0x00, 0x7A, 0x68, 0x13, 0x44,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xDA, 0x08,
    0x2A, 0x4B, 0x13, 0x44, 0x1A, 0x46, 0x3B, 0x68,
    0xDB, 0x00, 0x79, 0x68, 0x0B, 0x44, 0x5B, 0x68,
    0x03, 0xF0, 0x01, 0x03, 0x00, 0x2B, 0x16, 0xD0,
    0x3B, 0x68, 0xDB, 0x00, 0x79, 0x68, 0x0B, 0x44,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xD9, 0x08,
    0x20, 0x4B, 0x0B, 0x44, 0x19, 0x68, 0x3B, 0x68,
    0xDB, 0x00, 0x78, 0x68, 0x03, 0x44, 0x1B, 0x68,
    0x03, 0xF0, 0x1F, 0x03, 0x01, 0x20, 0x00, 0xFA,
    0x03, 0xF3, 0x0B, 0x43, 0x16, 0xE0, 0x3B, 0x68,
    0xDB, 0x00, 0x79, 0x68, 0x0B, 0x44, 0x1B, 0x68,
    0x03, 0xF0, 0x20, 0x03, 0xD9, 0x08, 0x15, 0x4B,
    0x0B, 0x44, 0x19, 0x68, 0x3B, 0x68, 0xDB, 0x00,
    0x78, 0x68, 0x03, 0x44, 0x1B, 0x68, 0x03, 0xF0,
    0x1F, 0x03, 0x01, 0x20, 0x00, 0xFA, 0x03, 0xF3,
    0xDB, 0x43, 0x0B, 0x40, 0x13, 0x60, 0x14, 0xE0,
    0x3B, 0x68, 0xDB, 0x00, 0x7A, 0x68, 0x13, 0x44,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xDA, 0x08,
    0x09, 0x4B, 0x13, 0x44, 0x3A, 0x68, 0xD2, 0x00,
    0x79, 0x68, 0x0A, 0x44, 0x12, 0x68, 0x02, 0xF0,
    0x1F, 0x02, 0x01, 0x21, 0x01, 0xFA, 0x02, 0xF2,
    0x1A, 0x60, 0x0C, 0x37, 0xBD, 0x46, 0x5D, 0xF8,
    0x04, 0x7B, 0x70, 0x47, 0x88, 0x00, 0x01, 0x40,
    0xA8, 0x00, 0x01, 0x40, 0x80, 0xB4, 0x83, 0xB0,
    0x00, 0xAF, 0x78, 0x60, 0x39, 0x60, 0x3B, 0x68,
    0xDB, 0x00, 0x7A, 0x68, 0x13, 0x44, 0x5B, 0x68,
    0x03, 0xF0, 0x02, 0x03, 0x00, 0x2B, 0x43, 0xD0,
    0x3B, 0x68, 0xDB, 0x00, 0x7A, 0x68, 0x13, 0x44,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xDA, 0x08,
    0x2A, 0x4B, 0x13, 0x44, 0x1A, 0x46, 0x3B, 0x68,
    0xDB, 0x00, 0x79, 0x68, 0x0B, 0x44, 0x5B, 0x68,
    0x03, 0xF0, 0x01, 0x03, 0x00, 0x2B, 0x16, 0xD1,
    0x3B, 0x68, 0xDB, 0x00, 0x79, 0x68, 0x0B, 0x44,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xD9, 0x08,
    0x20, 0x4B, 0x0B, 0x44, 0x19, 0x68, 0x3B, 0x68,
    0xDB, 0x00, 0x78, 0x68, 0x03, 0x44, 0x1B, 0x68,
    0x03, 0xF0, 0x1F, 0x03, 0x01, 0x20, 0x00, 0xFA,
    0x03, 0xF3, 0x0B, 0x43, 0x16, 0xE0, 0x3B, 0x68,
    0xDB, 0x00, 0x79, 0x68, 0x0B, 0x44, 0x1B, 0x68,
    0x03, 0xF0, 0x20, 0x03, 0xD9, 0x08, 0x15, 0x4B,
    0x0B, 0x44, 0x19, 0x68, 0x3B, 0x68, 0xDB, 0x00,
    0x78, 0x68, 0x03, 0x44, 0x1B, 0x68, 0x03, 0xF0,
    0x1F, 0x03, 0x01, 0x20, 0x00, 0xFA, 0x03, 0xF3,
    0xDB, 0x43, 0x0B, 0x40, 0x13, 0x60, 0x14, 0xE0,
    0x3B, 0x68, 0xDB, 0x00, 0x7A, 0x68, 0x13, 0x44,
    0x1B, 0x68, 0x03, 0xF0, 0x20, 0x03, 0xDA, 0x08,
    0x09, 0x4B, 0x13, 0x44, 0x3A, 0x68, 0xD2, 0x00,
    0x79, 0x68, 0x0A, 0x44, 0x12, 0x68, 0x02, 0xF0,
    0x1F, 0x02, 0x01, 0x21, 0x01, 0xFA, 0x02, 0xF2,
    0x1A, 0x60, 0x0C, 0x37, 0xBD, 0x46, 0x5D, 0xF8,
    0x04, 0x7B, 0x70, 0x47, 0x88, 0x00, 0x01, 0x40,
    0xB4, 0x00, 0x01, 0x40, 0x80, 0xB5, 0x86, 0xB0,
    0x00, 0xAF, 0xF8, 0x60, 0xB9, 0x60, 0x7A, 0x60,
    0x00, 0x23, 0x7B, 0x61, 0x14, 0xE0, 0x7B, 0x69,
    0x01, 0x22, 0x02, 0xFA, 0x03, 0xF3, 0x1A, 0x46,
    0x7B, 0x68, 0x13, 0x40, 0x00, 0x2B, 0x04, 0xD0,
    0xF8, 0x68, 0x79, 0x69, 0xFF, 0xF7, 0x0A, 0xFF,
    0x03, 0xE0, 0xF8, 0x68, 0x79, 0x69, 0xFF, 0xF7,
    0x75, 0xFF, 0x7B, 0x69, 0x01, 0x33, 0x7B, 0x61,
    0x7A, 0x69, 0xBB, 0x68, 0x9A, 0x42, 0xE6, 0xD3,
    0x18, 0x37, 0xBD, 0x46, 0x80, 0xBD, 0x00, 0xBF,
    0x12, 0x48, 0x13, 0x49, 0x01, 0x60, 0xD1, 0xF8,
    0x00, 0xD0, 0x12, 0x48, 0x01, 0x68, 0x41, 0xF4,
    0x70, 0x01, 0x01, 0x60, 0xBF, 0xF3, 0x4F, 0x8F,
    0xBF, 0xF3, 0x6F, 0x8F, 0x0E, 0x48, 0x0F, 0x49,
    0x0F, 0x4A, 0x50, 0xF8, 0x04, 0x3B, 0x41, 0xF8,
    0x04, 0x3B, 0x91, 0x42, 0x7F, 0xF7, 0xF9, 0xAF,
    0x0C, 0x48, 0x0D, 0x49, 0x4F, 0xF0, 0x00, 0x02,
    0x88, 0x42, 0xB8, 0xBF, 0x40, 0xF8, 0x04, 0x2B,
    0xFF, 0xF6, 0xFA, 0xAF, 0xFE, 0xF7, 0x0C, 0xFE,
    0x00, 0xBE, 0x00, 0x00, 0x08, 0xED, 0x00, 0xE0,
    0x00, 0x80, 0x00, 0x00, 0x88, 0xED, 0x00, 0xE0,
    0x5C, 0x9D, 0x00, 0x00, 0x00, 0x02, 0x00, 0x10,
    0x7C, 0x02, 0x00, 0x10, 0x7C, 0x02, 0x00, 0x10,
    0x8C, 0x03, 0x00, 0x10, 0x80, 0xB4, 0x00, 0xAF,
    0xFE, 0xE7, 0x00, 0xBF, 0x80, 0xB4, 0x00, 0xAF,
    0xFE, 0xE7, 0x00, 0xBF, 0x08, 0xB5, 0x00, 0xF0,
    0x33, 0xFA, 0x00, 0xF0, 0xD7, 0xFA, 0x00, 0xF0,
    0x67, 0xFB, 0x01, 0x20, 0x00, 0xF0, 0xDE, 0xFA,
    0x01, 0x20, 0x00, 0xF0, 0x87, 0xF8, 0xBD, 0xE8,
    0x08, 0x40, 0x00, 0xF0, 0xE5, 0xBA, 0x00, 0xBF,
    0x0D, 0x4B, 0x1B, 0x68, 0x10, 0xB5, 0x13, 0xF0,
    0x01, 0x04, 0x14, 0xD1, 0x0B, 0x4B, 0x0C, 0x4A,
    0x1A, 0x60, 0x00, 0xF0, 0x15, 0xFB, 0x0B, 0x4B,
    0x0B, 0x49, 0x0C, 0x4A, 0x73, 0x20, 0x18, 0x60,
    0x08, 0x68, 0x20, 0xF0, 0x70, 0x00, 0x08, 0x60,
    0x11, 0x68, 0x21, 0xF4, 0x7F, 0x41, 0x41, 0xF4,
    0x80, 0x51, 0x11, 0x60, 0x1C, 0x60, 0x10, 0xBD,
    0x84, 0x03, 0x00, 0x10, 0x80, 0x0E, 0x00, 0xE0,
    0x11, 0x05, 0x15, 0x00, 0x60, 0x00, 0x01, 0x40,
    0x54, 0x00, 0x01, 0x40, 0x28, 0x00, 0x01, 0x40,
    0x0F, 0x4B, 0x1B, 0x68, 0x10, 0xB5, 0x13, 0xF0,
    0x01, 0x04, 0x00, 0xD0, 0x10, 0xBD, 0x00, 0xF0,
    0xA1, 0xF9, 0x0C, 0x4B, 0x0C, 0x49, 0x0D, 0x4A,
    0x73, 0x20, 0x18, 0x60, 0x08, 0x68, 0x20, 0xF0,
    0x70, 0x00, 0x40, 0xF0, 0x20, 0x00, 0x08, 0x60,
    0x11, 0x68, 0x21, 0xF4, 0x7F, 0x41, 0x41, 0xF4,
    0xC0, 0x51, 0x11, 0x60, 0x1C, 0x60, 0xBD, 0xE8,
    0x10, 0x40, 0x00, 0xF0, 0x0F, 0xBB, 0x00, 0xBF,
    0x84, 0x03, 0x00, 0x10, 0x60, 0x00, 0x01, 0x40,
    0x54, 0x00, 0x01, 0x40, 0x28, 0x00, 0x01, 0x40,
    0x00, 0xF0, 0x96, 0xB9, 0x10, 0xB5, 0x04, 0x46,
    0x20, 0xB1, 0x06, 0x48, 0x06, 0x4A, 0x35, 0x21,
    0x00, 0xF0, 0xFE, 0xF8, 0x05, 0x4B, 0x06, 0x49,
    0x47, 0x20, 0x00, 0x22, 0x18, 0x60, 0x0C, 0x60,
    0x1A, 0x60, 0x10, 0xBD, 0x0C, 0x9D, 0x00, 0x00,
    0x24, 0x9D, 0x00, 0x00, 0x14, 0x40, 0x00, 0x40,
    0x18, 0x40, 0x00, 0x40, 0x04, 0x4B, 0x05, 0x48,
    0x1A, 0x68, 0x05, 0x4B, 0x12, 0xF0, 0x01, 0x0F,
    0x08, 0xBF, 0x18, 0x46, 0x70, 0x47, 0x00, 0xBF,
    0x18, 0x40, 0x00, 0x40, 0x00, 0x36, 0x6E, 0x01,
    0x00, 0x6C, 0xDC, 0x02, 0x02, 0x4B, 0x1A, 0x68,
    0x22, 0xEA, 0x00, 0x00, 0x18, 0x60, 0x70, 0x47,
    0x0C, 0x40, 0x00, 0x40, 0x02, 0x4B, 0x1A, 0x68,
    0x10, 0x43, 0x18, 0x60, 0x70, 0x47, 0x00, 0xBF,
    0x0C, 0x40, 0x00, 0x40, 0x30, 0xB4, 0x0C, 0x68,
    0x8D, 0x68, 0x49, 0x68, 0x29, 0x4A, 0x2A, 0x4B,
    0x00, 0x2C, 0x0C, 0xBF, 0x00, 0x24, 0x4F, 0xF0,
    0x00, 0x44, 0x00, 0x01, 0x41, 0xEA, 0x05, 0x41,
    0x21, 0x43, 0x81, 0x50, 0x19, 0x68, 0x01, 0xF0,
    0x0F, 0x01, 0x01, 0x29, 0x3A, 0xD0, 0x1B, 0x68,
    0x03, 0xF0, 0x0F, 0x03, 0x02, 0x2B, 0x35, 0xD0,
    0x13, 0x68, 0xC3, 0xF3, 0x44, 0x02, 0x01, 0x3A,
    0x04, 0x2A, 0xC3, 0xF3, 0x44, 0x43, 0x2D, 0xD9,
    0x01, 0x3B, 0x04, 0x2B, 0x2A, 0xD9, 0x1B, 0x4B,
    0x1B, 0x68, 0xC3, 0xF3, 0x44, 0x02, 0x01, 0x3A,
    0x04, 0x2A, 0xC3, 0xF3, 0x44, 0x43, 0x21, 0xD9,
    0x01, 0x3B, 0x04, 0x2B, 0x1E, 0xD9, 0x16, 0x4B,
    0x1B, 0x68, 0xC3, 0xF3, 0x44, 0x02, 0x01, 0x3A,
    0x04, 0x2A, 0xC3, 0xF3, 0x44, 0x43, 0x15, 0xD9,
    0x01, 0x3B, 0x04, 0x2B, 0x12, 0xD9, 0x11, 0x4B,
    0x1B, 0x68, 0xC3, 0xF3, 0x44, 0x02, 0x01, 0x3A,
    0x04, 0x2A, 0xC3, 0xF3, 0x44, 0x43, 0x09, 0xD9,
    0x01, 0x3B, 0x04, 0x2B, 0x06, 0xD9, 0x0C, 0x4B,
    0x1A, 0x68, 0x42, 0xF0, 0x04, 0x02, 0x1A, 0x60,
    0x30, 0xBC, 0x70, 0x47, 0x08, 0x4B, 0x1A, 0x68,
    0x22, 0xF0, 0x04, 0x02, 0x1A, 0x60, 0x30, 0xBC,
    0x70, 0x47, 0x00, 0xBF, 0x0C, 0x80, 0x00, 0x40,
    0x00, 0x81, 0x00, 0x40, 0x1C, 0x80, 0x00, 0x40,
    0x2C, 0x80, 0x00, 0x40, 0x3C, 0x80, 0x00, 0x40,
    0x20, 0x10, 0x02, 0x40, 0x00, 0x01, 0x07, 0x4B,
    0xC2, 0x58, 0x10, 0xB4, 0x01, 0xF0, 0x08, 0x24,
    0x22, 0xEA, 0x04, 0x02, 0x01, 0xF0, 0x01, 0x11,
    0x11, 0x43, 0xC1, 0x50, 0x5D, 0xF8, 0x04, 0x4B,
    0x70, 0x47, 0x00, 0xBF, 0x0C, 0x80, 0x00, 0x40,
    0x00, 0x01, 0x03, 0x4B, 0xC2, 0x58, 0x01, 0xF0,
    0x08, 0x21, 0x0A, 0x43, 0xC2, 0x50, 0x70, 0x47,
    0x0C, 0x80, 0x00, 0x40, 0xF0, 0xB4, 0x00, 0x01,
    0x1B, 0x4C, 0x04, 0x59, 0x11, 0xF5, 0x80, 0x3F,
    0x08, 0xBF, 0x24, 0x0C, 0x04, 0xF4, 0xC0, 0x74,
    0x80, 0x2C, 0x16, 0xD0, 0x4F, 0xF6, 0xFF, 0x75,
    0x00, 0x24, 0xA9, 0x42, 0x26, 0x46, 0x15, 0x4F,
    0xDF, 0xF8, 0x54, 0xC0, 0x19, 0xD0, 0x11, 0xF5,
    0x80, 0x3F, 0x1B, 0xD0, 0x33, 0x0C, 0x91, 0xB2,
    0x1B, 0x04, 0x0C, 0x43, 0x43, 0xEA, 0x12, 0x42,
    0xC4, 0x51, 0x40, 0xF8, 0x0C, 0x20, 0xF0, 0xBC,
    0x70, 0x47, 0x4F, 0xF6, 0xFF, 0x75, 0xA9, 0x42,
    0x4F, 0xEA, 0x02, 0x44, 0x16, 0x46, 0x09, 0x4F,
    0xDF, 0xF8, 0x24, 0xC0, 0xC3, 0xEB, 0x02, 0x02,
    0xE5, 0xD1, 0x92, 0xB2, 0x14, 0x43, 0xC4, 0x51,
    0xF0, 0xBC, 0x70, 0x47, 0x92, 0xB2, 0x14, 0x43,
    0x40, 0xF8, 0x0C, 0x40, 0xF0, 0xBC, 0x70, 0x47,
    0x0C, 0x80, 0x00, 0x40, 0x04, 0x80, 0x00, 0x40,
    0x08, 0x80, 0x00, 0x40, 0x02, 0x4B, 0x1A, 0x68,
    0x10, 0x43, 0x18, 0x60, 0x70, 0x47, 0x00, 0xBF,
    0x00, 0x82, 0x00, 0x40, 0x01, 0x4B, 0x18, 0x60,
    0x70, 0x47, 0x00, 0xBF, 0x08, 0x82, 0x00, 0x40,
    0xFE, 0xE7, 0x00, 0xBF, 0x08, 0xB5, 0x02, 0x4B,
    0x5B, 0x6B, 0x98, 0x47, 0x08, 0xBD, 0x00, 0xBF,
    0x2C, 0x02, 0x00, 0x10, 0x0F, 0x28, 0x0B, 0xD8,
    0x05, 0x28, 0x12, 0xD0, 0x06, 0x28, 0x16, 0xD0,
    0x04, 0x28, 0x04, 0xD1, 0x0D, 0x4B, 0x1A, 0x68,
    0x42, 0xF4, 0x80, 0x32, 0x1A, 0x60, 0x70, 0x47,
    0x10, 0x38, 0x00, 0xF0, 0x1F, 0x00, 0x01, 0x22,
    0x09, 0x4B, 0x02, 0xFA, 0x00, 0xF0, 0x18, 0x60,
    0x70, 0x47, 0x06, 0x4B, 0x1A, 0x68, 0x42, 0xF4,
    0x00, 0x32, 0x1A, 0x60, 0x70, 0x47, 0x03, 0x4B,
    0x1A, 0x68, 0x42, 0xF4, 0x80, 0x22, 0x1A, 0x60,
    0x70, 0x47, 0x00, 0xBF, 0x24, 0xED, 0x00, 0xE0,
    0x00, 0xE1, 0x00, 0xE0, 0xEF, 0xF3, 0x10, 0x80,
    0x62, 0xB6, 0x70, 0x47, 0xEF, 0xF3, 0x10, 0x80,
    0x72, 0xB6, 0x70, 0x47, 0x80, 0xF3, 0x10, 0x88,
    0x70, 0x47, 0x00, 0xBF, 0x10, 0xB5, 0x04, 0x46,
    0xFF, 0xF7, 0xD0, 0xFE, 0x04, 0x4B, 0xA3, 0xFB,
    0x00, 0x20, 0x80, 0x0C, 0x04, 0xFB, 0x00, 0xF0,
    0xBD, 0xE8, 0x10, 0x40, 0xFF, 0xF7, 0xB6, 0xBF,
    0x81, 0x9F, 0x5E, 0x16, 0x0D, 0x4B, 0x1B, 0x68,
    0xDA, 0x07, 0x30, 0xB4, 0x14, 0xD4, 0x0C, 0x4A,
    0x13, 0x68, 0x43, 0xF0, 0x80, 0x73, 0x13, 0x60,
    0x13, 0x68, 0xDB, 0x01, 0xFC, 0xD5, 0x09, 0x48,
    0x09, 0x4D, 0x0A, 0x4C, 0x0A, 0x49, 0x0B, 0x4B,
    0x05, 0x60, 0x0B, 0x4A, 0x0F, 0x25, 0x4F, 0xF0,
    0xFF, 0x30, 0x25, 0x60, 0x08, 0x60, 0x1A, 0x60,
    0x30, 0xBC, 0x70, 0x47, 0x84, 0x03, 0x00, 0x10,
    0xFC, 0xED, 0x00, 0xE0, 0xB0, 0x0F, 0x00, 0xE0,
    0x55, 0xCE, 0xAC, 0xC5, 0x40, 0x0E, 0x00, 0xE0,
    0x00, 0x0E, 0x00, 0xE0, 0x80, 0x0E, 0x00, 0xE0,
    0x11, 0x05, 0x15, 0x00, 0x08, 0xB5, 0x08, 0x4A,
    0x13, 0x68, 0x1B, 0x02, 0xFC, 0xD4, 0xFF, 0xF7,
    0x91, 0xFE, 0x06, 0x4B, 0xA3, 0xFB, 0x00, 0x20,
    0x80, 0x0C, 0x32, 0x23, 0x03, 0xFB, 0x00, 0xF0,
    0xBD, 0xE8, 0x08, 0x40, 0xFF, 0xF7, 0x76, 0xBF,
    0x80, 0x0E, 0x00, 0xE0, 0x81, 0x9F, 0x5E, 0x16,
    0x03, 0x78, 0x30, 0xB4, 0x93, 0xB1, 0x02, 0x46,
    0x00, 0x23, 0x12, 0xF8, 0x01, 0x1F, 0x01, 0x33,
    0x00, 0x29, 0xFA, 0xD1, 0x4F, 0xF0, 0x60, 0x42,
    0xC5, 0x18, 0x14, 0x46, 0x10, 0xF8, 0x01, 0x1B,
    0x13, 0x68, 0x00, 0x2B, 0xFC, 0xD0, 0xA8, 0x42,
    0x21, 0x70, 0xF7, 0xD1, 0x30, 0xBC, 0x70, 0x47,
    0x0D, 0x4B, 0x0E, 0x49, 0x1B, 0x68, 0x0E, 0x4A,
    0x09, 0x68, 0xC1, 0x60, 0x70, 0xB4, 0x41, 0x68,
    0x15, 0x68, 0x0C, 0x4C, 0x42, 0x69, 0x29, 0x43,
    0x41, 0x60, 0x24, 0x68, 0x03, 0xF0, 0x01, 0x06,
    0xC3, 0xF3, 0x40, 0x01, 0x22, 0x43, 0xC3, 0xF3,
    0x80, 0x03, 0x06, 0x70, 0x01, 0x72, 0x03, 0x74,
    0x42, 0x61, 0x70, 0xBC, 0x70, 0x47, 0x00, 0xBF,
    0xCC, 0x01, 0x02, 0x40, 0xC4, 0x01, 0x02, 0x40,
    0xC0, 0x01, 0x02, 0x40, 0xC8, 0x01, 0x02, 0x40,
    0x21, 0x4B, 0x1A, 0x68, 0x92, 0x07, 0xF0, 0xB4,
    0x02, 0xD5, 0x1B, 0x68, 0xDB, 0x07, 0x35, 0xD4,
    0x1E, 0x4B, 0x1B, 0x68, 0x03, 0xF0, 0xF0, 0x03,
    0x10, 0x2B, 0x31, 0xD0, 0x1C, 0x4B, 0x1A, 0x4A,
    0x19, 0x68, 0x41, 0xF0, 0x02, 0x01, 0x19, 0x60,
    0x19, 0x68, 0x41, 0xF0, 0x01, 0x01, 0x19, 0x60,
    0x13, 0x68, 0x03, 0xF0, 0x03, 0x03, 0x03, 0x2B,
    0xFA, 0xD1, 0x14, 0x4B, 0x1B, 0x68, 0x03, 0xF0,
    0xF0, 0x03, 0x10, 0x2B, 0x1A, 0xD1, 0x13, 0x4B,
    0x13, 0x49, 0x14, 0x4A, 0x1F, 0x68, 0x14, 0x48,
    0x1E, 0x68, 0x0B, 0x68, 0x14, 0x68, 0x05, 0x68,
    0x23, 0xF4, 0x7F, 0x73, 0xC7, 0xF3, 0x09, 0x07,
    0xC6, 0xF3, 0x09, 0x46, 0x23, 0xF0, 0x03, 0x03,
    0x24, 0xF4, 0x7C, 0x34, 0x3B, 0x43, 0x44, 0xEA,
    0x06, 0x34, 0x25, 0xF0, 0x01, 0x05, 0x05, 0x60,
    0x0B, 0x60, 0x14, 0x60, 0xF0, 0xBC, 0x70, 0x47,
    0x07, 0x4B, 0xC1, 0x22, 0x1A, 0x60, 0xC9, 0xE7,
    0x04, 0x10, 0x02, 0x40, 0x0C, 0x00, 0x02, 0x40,
    0x00, 0x10, 0x02, 0x40, 0x00, 0x30, 0x02, 0x50,
    0x80, 0x00, 0x02, 0x40, 0x88, 0x00, 0x02, 0x40,
    0x60, 0x00, 0x02, 0x40, 0x23, 0x4B, 0x1A, 0x68,
    0x91, 0x07, 0x70, 0xB4, 0x02, 0xD4, 0x1B, 0x68,
    0xDA, 0x07, 0x12, 0xD5, 0x20, 0x4B, 0x1B, 0x68,
    0x03, 0xF0, 0xF0, 0x03, 0x10, 0x2B, 0x0E, 0xD0,
    0x1E, 0x4B, 0x1C, 0x4A, 0x19, 0x68, 0x21, 0xF0,
    0x02, 0x01, 0x19, 0x60, 0x19, 0x68, 0x21, 0xF0,
    0x01, 0x01, 0x19, 0x60, 0x13, 0x68, 0x9B, 0x07,
    0xFC, 0xD1, 0x70, 0xBC, 0x70, 0x47, 0x18, 0x4B,
    0x18, 0x4A, 0x19, 0x48, 0x13, 0x49, 0x1D, 0x68,
    0x1C, 0x68, 0x16, 0x68, 0x00, 0x68, 0x0B, 0x68,
    0x9B, 0x07, 0x0D, 0xD5, 0x26, 0xF4, 0x7F, 0x73,
    0x51, 0xF8, 0x04, 0x6C, 0xC5, 0xF3, 0x09, 0x05,
    0x23, 0xF0, 0x03, 0x03, 0x2B, 0x43, 0x26, 0xF0,
    0x02, 0x05, 0x41, 0xF8, 0x04, 0x5C, 0x13, 0x60,
    0x08, 0x4B, 0x1B, 0x68, 0xDA, 0x07, 0xD3, 0xD5,
    0x08, 0x4B, 0x0B, 0x4A, 0x1D, 0x68, 0x20, 0xF4,
    0x7C, 0x30, 0xC4, 0xF3, 0x09, 0x41, 0x40, 0xEA,
    0x01, 0x31, 0x25, 0xF0, 0x01, 0x00, 0x18, 0x60,
    0x11, 0x60, 0xC5, 0xE7, 0x04, 0x10, 0x02, 0x40,
    0x0C, 0x00, 0x02, 0x40, 0x00, 0x10, 0x02, 0x40,
    0x04, 0x30, 0x02, 0x50, 0x80, 0x00, 0x02, 0x40,
    0x88, 0x00, 0x02, 0x40, 0x04, 0x4B, 0x1A, 0x68,
    0x42, 0xF0, 0x06, 0x02, 0x1A, 0x60, 0x1A, 0x68,
    0x22, 0xF0, 0x04, 0x02, 0x1A, 0x60, 0x70, 0x47,
    0x18, 0x10, 0x02, 0x40, 0x05, 0x4B, 0x1A, 0x68,
    0x18, 0xB9, 0x22, 0xF0, 0x80, 0x02, 0x1A, 0x60,
    0x70, 0x47, 0x42, 0xF0, 0x80, 0x02, 0x1A, 0x60,
    0x70, 0x47, 0x00, 0xBF, 0x0C, 0x40, 0x00, 0x40,
    0x02, 0x4B, 0x1A, 0x68, 0x42, 0xF0, 0x10, 0x02,
    0x1A, 0x60, 0x70, 0x47, 0x50, 0x40, 0x00, 0x40,
    0x30, 0xB5, 0x04, 0x46, 0x83, 0xB0, 0xFF, 0xF7,
    0x9D, 0xFE, 0x1C, 0xB1, 0x13, 0x4B, 0x1B, 0x68,
    0xDB, 0x07, 0x0A, 0xD5, 0x12, 0x4B, 0x1A, 0x68,
    0x22, 0xF0, 0x04, 0x02, 0x1A, 0x60, 0x30, 0xBF,
    0x03, 0xB0, 0xBD, 0xE8, 0x30, 0x40, 0xFF, 0xF7,
    0x91, 0xBE, 0x0D, 0x4A, 0x0D, 0x4B, 0x11, 0x68,
    0x41, 0xF0, 0x04, 0x01, 0x11, 0x60, 0x1C, 0x68,
    0x1B, 0x68, 0x01, 0x90, 0x03, 0xF0, 0x01, 0x05,
    0xFF, 0xF7, 0x68, 0xFF, 0x30, 0xBF, 0x14, 0xF0,
    0x02, 0x0F, 0x01, 0x98, 0x01, 0xD1, 0x00, 0x2D,
    0xE6, 0xD0, 0x01, 0x90, 0xFF, 0xF7, 0x0C, 0xFF,
    0x01, 0x98, 0xE1, 0xE7, 0x50, 0x02, 0x02, 0x40,
    0x10, 0xED, 0x00, 0xE0, 0x00, 0x10, 0x02, 0x40,
    0x38, 0xB5, 0x12, 0x49, 0x12, 0x4B, 0x00, 0x20,
    0x01, 0x22, 0x08, 0x60, 0x1A, 0x60, 0xFF, 0xF7,
    0x41, 0xFD, 0x10, 0x4B, 0x98, 0x42, 0x15, 0xD0,
    0x0F, 0x4B, 0x98, 0x42, 0x0E, 0xD0, 0x0F, 0x48,
    0x0F, 0x4B, 0x10, 0x49, 0x02, 0x25, 0x00, 0x24,
    0x40, 0xF2, 0x01, 0x22, 0x05, 0x60, 0x0C, 0x60,
    0x32, 0x20, 0x1A, 0x60, 0xBD, 0xE8, 0x38, 0x40,
    0xFF, 0xF7, 0x58, 0xBE, 0x0A, 0x4B, 0x02, 0x22,
    0x1A, 0x60, 0xEC, 0xE7, 0x08, 0x4B, 0x05, 0x22,
    0x1A, 0x60, 0xE8, 0xE7, 0x04, 0x03, 0x04, 0xE0,
    0x04, 0x00, 0x04, 0xE0, 0x00, 0x6C, 0xDC, 0x02,
    0x00, 0x36, 0x6E, 0x01, 0xF0, 0x00, 0x04, 0xE0,
    0x50, 0x02, 0x02, 0x40, 0x00, 0x0F, 0x04, 0xE0,
    0x10, 0x00, 0x04, 0xE0, 0x01, 0x4B, 0x00, 0x22,
    0x1A, 0x60, 0x70, 0x47, 0x50, 0x02, 0x02, 0x40,
    0x01, 0x4B, 0x37, 0x22, 0x1A, 0x60, 0x70, 0x47,
    0x08, 0xC0, 0x00, 0x40, 0x4F, 0xEA, 0x41, 0x02,
    0xB2, 0xF1, 0xE0, 0x43, 0x24, 0xBF, 0xB3, 0xF5,
    0x00, 0x1C, 0xDC, 0xF1, 0xFE, 0x5C, 0x0D, 0xD9,
    0x01, 0xF0, 0x00, 0x4C, 0x4F, 0xEA, 0xC0, 0x02,
    0x4C, 0xEA, 0x50, 0x70, 0xB2, 0xF1, 0x00, 0x4F,
    0x40, 0xEB, 0x83, 0x00, 0x08, 0xBF, 0x20, 0xF0,
    0x01, 0x00, 0x70, 0x47, 0x11, 0xF0, 0x80, 0x4F,
    0x21, 0xD1, 0x13, 0xF1, 0x38, 0x72, 0xBC, 0xBF,
    0x01, 0xF0, 0x00, 0x40, 0x70, 0x47, 0x41, 0xF4,
    0x80, 0x11, 0x4F, 0xEA, 0x52, 0x52, 0xC2, 0xF1,
    0x18, 0x02, 0xC2, 0xF1, 0x20, 0x0C, 0x10, 0xFA,
    0x0C, 0xF3, 0x20, 0xFA, 0x02, 0xF0, 0x18, 0xBF,
    0x40, 0xF0, 0x01, 0x00, 0x4F, 0xEA, 0xC1, 0x23,
    0x4F, 0xEA, 0xD3, 0x23, 0x03, 0xFA, 0x0C, 0xFC,
    0x40, 0xEA, 0x0C, 0x00, 0x23, 0xFA, 0x02, 0xF3,
    0x4F, 0xEA, 0x43, 0x03, 0xCC, 0xE7, 0x7F, 0xEA,
    0x62, 0x53, 0x07, 0xD1, 0x50, 0xEA, 0x01, 0x33,
    0x1E, 0xBF, 0x4F, 0xF0, 0xFE, 0x40, 0x40, 0xF4,
    0x40, 0x00, 0x70, 0x47, 0x01, 0xF0, 0x00, 0x40,
    0x40, 0xF0, 0xFE, 0x40, 0x40, 0xF4, 0x00, 0x00,
    0x70, 0x47, 0x00, 0xBF, 0x42, 0x69, 0x6E, 0x61,
    0x72, 0x79, 0x20, 0x43, 0x6F, 0x75, 0x6E, 0x74,
    0x65, 0x72, 0x20, 0x45, 0x78, 0x61, 0x6D, 0x70,
    0x6C, 0x65, 0x0A, 0x00, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x00, 0x00, 0x1B, 0x5B, 0x32, 0x4A,
    0x1B, 0x5B, 0x48, 0x00, 0x2E, 0x2E, 0x2F, 0x2E,
    0x2F, 0x61, 0x6D, 0x5F, 0x68, 0x61, 0x6C, 0x5F,
    0x63, 0x6C, 0x6B, 0x67, 0x65, 0x6E, 0x2E, 0x63,
    0x00, 0x00, 0x00, 0x00, 0x61, 0x6D, 0x5F, 0x68,
    0x61, 0x6C, 0x5F, 0x63, 0x6C, 0x6B, 0x67, 0x65,
    0x6E, 0x5F, 0x73, 0x79, 0x73, 0x63, 0x6C, 0x6B,
    0x5F, 0x73, 0x65, 0x6C, 0x65, 0x63, 0x74, 0x28,
    0x29, 0x3A, 0x20, 0x69, 0x6E, 0x76, 0x61, 0x6C,
    0x69, 0x64, 0x20, 0x63, 0x6C, 0x6F, 0x63, 0x6B,
    0x20, 0x73, 0x65, 0x74, 0x74, 0x69, 0x6E, 0x67,
    0x2E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x56, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x2E, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x2F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x31, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4D, 0x00, 0x00, 0x08, 0x51, 0x00, 0x00, 0x08,
    0x55, 0x00, 0x00, 0x08, 0x59, 0x00, 0x00, 0x08,
    0x6D, 0x00, 0x00, 0x08, 0x71, 0x00, 0x00, 0x08,
    0x7D, 0x00, 0x00, 0x08, 0x81, 0x00, 0x00, 0x08,
    0x89, 0x00, 0x00, 0x08, 0x91, 0x00, 0x00, 0x08,
    0x99, 0x00, 0x00, 0x08, 0x75, 0x00, 0x00, 0x08,
    0x79, 0x00, 0x00, 0x08, 0x9D, 0x00, 0x00, 0x08,
    0x5D, 0x00, 0x00, 0x08, 0x61, 0x00, 0x00, 0x08,
    0x65, 0x00, 0x00, 0x08, 0x69, 0x00, 0x00, 0x08,
    0x85, 0x00, 0x00, 0x08, 0x8D, 0x00, 0x00, 0x08
};

typedef struct
{
    uint8_t ui8Tag;
    uint8_t ui8Length;
}sHeader;

typedef enum
{
    IOM_DATA_TAG,
    IOM_DATA_LENGTH,
    IOM_DATA_VALUE
}eIOM_Data;

//*****************************************************************************
//
// Global message buffer for the IO master.
//
//*****************************************************************************
volatile bool bIosInt = false;
#define AM_IOS_LRAM_SIZE_MAX        0x78
uint8_t g_pRefData[AM_IOS_LRAM_SIZE_MAX];
uint8_t g_pIomRecvBuf[AM_IOS_LRAM_SIZE_MAX];
#define AM_IOS_HEADER_SIZE          sizeof(sHeader)
#define AM_IOS_MAX_DATA_SIZE        (AM_IOS_LRAM_SIZE_MAX - AM_IOS_HEADER_SIZE)

void *g_IOMHandle;
volatile bool g_IOMDMAComplete = false;

//*****************************************************************************
//
// Configuration structure for the IO Master.
//
//*****************************************************************************
uint32_t        DMATCBBuffer[1024];

static am_hal_iom_config_t g_sIOMSpiConfig =
{
    .eInterfaceMode = AM_HAL_IOM_SPI_MODE,
//    .ui32ClockFreq = AM_HAL_IOM_12MHZ,
//    .ui32ClockFreq = AM_HAL_IOM_8MHZ,
//    .ui32ClockFreq = AM_HAL_IOM_6MHZ,
    .ui32ClockFreq = AM_HAL_IOM_4MHZ,
//    .ui32ClockFreq = AM_HAL_IOM_3MHZ,
//    .ui32ClockFreq = AM_HAL_IOM_2MHZ,
//    .ui32ClockFreq = AM_HAL_IOM_1_5MHZ,
//    .ui32ClockFreq = AM_HAL_IOM_1MHZ,
//    .ui32ClockFreq = AM_HAL_IOM_750KHZ,
//    .ui32ClockFreq = AM_HAL_IOM_500KHZ,
//    .ui32ClockFreq = AM_HAL_IOM_400KHZ,
//    .ui32ClockFreq = AM_HAL_IOM_375KHZ,
//    .ui32ClockFreq = AM_HAL_IOM_250KHZ,
//    .ui32ClockFreq = AM_HAL_IOM_100KHZ,
//    .ui32ClockFreq = AM_HAL_IOM_50KHZ,
//    .ui32ClockFreq = AM_HAL_IOM_10KHZ,
    .eSpiMode = AM_HAL_IOM_SPI_MODE_0,
    .pNBTxnBuf          = &DMATCBBuffer[0],
    .ui32NBTxnBufLength = sizeof(DMATCBBuffer) / 4
};

static am_hal_iom_config_t g_sIOMI2cConfig =
{
    .eInterfaceMode = AM_HAL_IOM_I2C_MODE,
    .ui32ClockFreq  = AM_HAL_IOM_1MHZ,
};

const am_hal_gpio_pincfg_t g_AM_BSP_GPIO_HANDSHAKE =
{
    .uFuncSel       = AM_HAL_PIN_10_GPIO,
    .eDriveStrength = AM_HAL_GPIO_PIN_DRIVESTRENGTH_2MA,
    .eIntDir        = AM_HAL_GPIO_PIN_INTDIR_LO2HI,
    .eGPInput       = AM_HAL_GPIO_PIN_INPUT_ENABLE,
};

const am_hal_gpio_pincfg_t g_AM_BSP_GPIO_DISABLE =
{
    .uFuncSel            = AM_HAL_PIN_4_GPIO,
    .eGPOutcfg           = AM_HAL_GPIO_PIN_OUTCFG_DISABLE,
    .eGPInput            = AM_HAL_GPIO_PIN_INPUT_NONE,
};

// ISR callback for the host IOINT
static void hostint_handler(void)
{
    bIosInt = true;
}

static void pfnIOM_LRAM_Callback(void *pCallbackCtxt, uint32_t status)
{
    // Set the DMA complete flag.
    g_IOMDMAComplete = true;
}

//*****************************************************************************
//
// Interrupt handler for the GPIO pins.
//
//*****************************************************************************
//
//! Take over default ISR for IOM 0. (Queue mode service)
//
void
am_iomaster0_isr(void)
{
    uint32_t ui32Status;

    if (!am_hal_iom_interrupt_status_get(g_IOMHandle, true, &ui32Status))
    {
        if ( ui32Status )
        {
            am_hal_iom_interrupt_clear(g_IOMHandle, ui32Status);
            am_hal_iom_interrupt_service(g_IOMHandle, ui32Status);
        }
    }
}

void am_gpio_isr(void)
{
    uint64_t ui64Status;

    //
    // Read and clear the GPIO interrupt status.
    //
    am_hal_gpio_interrupt_status_get(false, &ui64Status);
    am_hal_gpio_interrupt_clear(ui64Status);
    am_hal_gpio_interrupt_service(ui64Status);
}

void iom_slave_read(bool bSpi, uint32_t offset, uint8_t *pBuf, uint32_t size)
{
    am_hal_iom_transfer_t       Transaction;

    Transaction.ui8Priority     = 1;        // High priority for now.
    Transaction.ui32InstrLen    = 1;
    Transaction.ui32Instr       = offset;
    Transaction.eDirection      = AM_HAL_IOM_RX;
    Transaction.ui32NumBytes    = size;
    Transaction.pui32RxBuffer   = (uint32_t *)pBuf;
    Transaction.bContinue       = false;
    Transaction.ui8RepeatCount  = 0;
    Transaction.ui32PauseCondition = 0;
    Transaction.ui32StatusSetClr = 0;

    if ( bSpi )
    {
        Transaction.uPeerInfo.ui32SpiChipSelect = AM_BSP_IOM0_CS_CHNL;
    }
    else
    {
        Transaction.uPeerInfo.ui32I2CDevAddr = I2C_ADDR;
    }
    g_IOMDMAComplete = false;
    am_hal_iom_nonblocking_transfer(g_IOMHandle, &Transaction, pfnIOM_LRAM_Callback, NULL);
    while (!g_IOMDMAComplete);
}

void iom_slave_write(bool bSpi, uint32_t offset, uint8_t *pBuf, uint32_t size)
{
    am_hal_iom_transfer_t       Transaction;

    Transaction.ui8Priority     = 1;        // High priority for now.
    Transaction.ui32InstrLen    = 1;
    Transaction.ui32Instr       = (AM_IOSTEST_WRITE_CMD_BIT | offset);
    Transaction.eDirection      = AM_HAL_IOM_TX;
    Transaction.ui32NumBytes    = size;
    Transaction.pui32TxBuffer   = (uint32_t *)pBuf;
    Transaction.bContinue       = false;
    Transaction.ui8RepeatCount  = 0;
    Transaction.ui32PauseCondition = 0;
    Transaction.ui32StatusSetClr = 0;

    if ( bSpi )
    {
        Transaction.uPeerInfo.ui32SpiChipSelect = AM_BSP_IOM0_CS_CHNL;
    }
    else
    {
        Transaction.uPeerInfo.ui32I2CDevAddr = I2C_ADDR;
    }
    g_IOMDMAComplete = false;
    am_hal_iom_nonblocking_transfer(g_IOMHandle, &Transaction, pfnIOM_LRAM_Callback, NULL);
    while (!g_IOMDMAComplete);
}

static void iom_set_up(uint32_t iomModule, bool bSpi)
{
    uint32_t ioIntEnable = HANDSHAKE_IOS_TO_IOM;

    //
    // Initialize the IOM.
    //
    am_hal_iom_initialize(iomModule, &g_IOMHandle);

    am_hal_iom_power_ctrl(g_IOMHandle, AM_HAL_SYSCTRL_WAKE, false);

    if ( bSpi )
    {
        //
        // Set the required configuration settings for the IOM.
        //
        am_hal_iom_configure(g_IOMHandle, &g_sIOMSpiConfig);

        //
        // Configure the IOM pins.
        //
        am_bsp_iom_pins_enable(iomModule, AM_HAL_IOM_SPI_MODE);
    }
    else
    {
        //
        // Set the required configuration settings for the IOM.
        //
        am_hal_iom_configure(g_IOMHandle, &g_sIOMI2cConfig);

        //
        // Configure the IOM pins.
        //
        am_bsp_iom_pins_enable(iomModule, AM_HAL_IOM_I2C_MODE);
    }

    // Enable interrupts for NB send to work
    am_hal_iom_interrupt_enable(g_IOMHandle, 0xFF);
    NVIC_EnableIRQ(IOMSTR0_IRQn);

    //
    // Enable the IOM.
    //
    am_hal_iom_enable(g_IOMHandle);
    am_hal_gpio_pinconfig(HANDSHAKE_PIN, g_AM_BSP_GPIO_HANDSHAKE);

    // Set up the host IO interrupt
    am_hal_gpio_interrupt_clear(AM_HAL_GPIO_BIT(HANDSHAKE_PIN));
    // Register handler for IOS => IOM interrupt
    am_hal_gpio_interrupt_register(HANDSHAKE_PIN, hostint_handler);
    am_hal_gpio_interrupt_enable(AM_HAL_GPIO_BIT(HANDSHAKE_PIN));
    NVIC_EnableIRQ(GPIO_IRQn);

    // Set up IOCTL interrupts
    // IOS ==> IOM
    iom_slave_write(bSpi, IOSOFFSET_INTEN, (uint8_t*)&ioIntEnable, 1);
}

static void iom_clean_up(uint32_t iomModule, bool bSpi)
{
    uint32_t ioIntDisable = 0;

    // Clean up IOCTL interrupts
    // IOS ==> IOM
    iom_slave_write(bSpi, IOSOFFSET_INTEN, (uint8_t*)&ioIntDisable, 1);

    //
    // Turn off the IOM for this operation.
    //
    am_hal_iom_disable(g_IOMHandle);

    if (bSpi)
    {
        //
        // Clean up IOM SPI pins. Attributes are set in am_bsp_gpio.h.
        //
        am_bsp_iom_pins_disable(iomModule, AM_HAL_IOM_SPI_MODE);
    }
    else
    {
        //
        // Clean up IOM I2C pins. Attributes are set in am_bsp_gpio.h.
        //
        am_bsp_iom_pins_disable(iomModule, AM_HAL_IOM_I2C_MODE);
    }

    // Clean up the host IO interrupt
    am_hal_gpio_state_write(HANDSHAKE_PIN, AM_HAL_GPIO_OUTPUT_CLEAR);
    am_hal_gpio_pinconfig(HANDSHAKE_PIN, g_AM_BSP_GPIO_DISABLE);
    am_hal_gpio_interrupt_disable(AM_HAL_GPIO_BIT(HANDSHAKE_PIN));
    NVIC_DisableIRQ(GPIO_IRQn);

    //
    // Disable power to IOM.
    //
    am_hal_iom_power_ctrl(g_IOMHandle, AM_HAL_SYSCTRL_DEEPSLEEP, false);
}

// Verify received data from IOS against the reference
static bool
verify_result(uint8_t length)
{
    if (g_pRefData[IOM_DATA_TAG] != g_pIomRecvBuf[IOM_DATA_TAG])
    {
        am_util_stdio_printf("Wrong Tag number!\n");
        return false;
    }

    if (g_pRefData[IOM_DATA_LENGTH] != g_pIomRecvBuf[IOM_DATA_LENGTH])
    {
        am_util_stdio_printf("Wrong data length!\n");
        return false;
    }
    for (uint8_t i = IOM_DATA_VALUE; i < length; i++)
    {
        if (g_pRefData[i] != (g_pIomRecvBuf[i] ^ TEST_XOR_BYTE))
        {
            am_util_stdio_printf("Buffer Validation failed @i=%d Rcvd 0x%x Expected 0x%x\n",
                i, g_pIomRecvBuf[i], g_pRefData[i] ^ TEST_XOR_BYTE);
            return false;
        }
    }
    return true;
}

//*****************************************************************************
//
// Main function.
//
//*****************************************************************************
int main(void)
{
    uint32_t iom = IOM_MODULE;
    bool bSpi = USE_SPI;
    uint8_t data;
    uint8_t ui8Resp = 0;
    uint32_t i = 0;
    uint32_t ui32Offset = 0;
    uint8_t ui8Tag = 1;
    uint8_t ui8Length = AM_IOS_LRAM_SIZE_MAX;

    am_hal_clkgen_control(AM_HAL_CLKGEN_CONTROL_SYSCLK_MAX, 0);

    //
    // Set the default cache configuration
    //
    am_hal_cachectrl_config(&am_hal_cachectrl_defaults);
    am_hal_cachectrl_enable();

    //
    // Configure the board for low power operation.
    //
    am_bsp_low_power_init();

    //
    // Enable the ITM print interface.
    //
    am_bsp_itm_printf_enable();

    //
    // Clear the terminal and print the banner.
    //
    am_util_stdio_terminal_clear();
    am_util_stdio_printf("IOS Test Host.\n");


    //
    // Allow time for all printing to finish.
    //
    am_util_delay_ms(10);

    // Initialize the Trigger GPIO.
    am_hal_gpio_pinconfig(IOM_TRIGGER_GPIO, g_AM_HAL_GPIO_OUTPUT);

    //
    // Enable Interrupts.
    //
    am_hal_interrupt_master_enable();

    //
    // Set up the IOM
    //
#if defined(MIKROE_1032) || defined(MIKROE_2529)
    g_sIOMI2cConfig.ui32ClockFreq = AM_HAL_IOM_400KHZ;
#endif

    iom_set_up(iom, bSpi);

    // Make sure the print is complete
    am_util_delay_ms(100);

    // Init buffer
    for (i = 0; i < AM_IOS_LRAM_SIZE_MAX; i++)
    {
        g_pIomRecvBuf[i] = INBUFFER_EMPTY;
        g_pRefData[i]    = INBUFFER_EMPTY;
    }

    // First handshake
    data = AM_IOSTEST_CMD_START_DATA;
    iom_slave_write(bSpi, IOSOFFSET_HANDSHAKE, &data, 1);

    while (1)
    {
        //
        // Disable interrupt while we decide whether we're going to sleep.
        //
        uint32_t ui32IntStatus = am_hal_interrupt_master_disable();

        if ( bIosInt == true )
        {
            //
            // Enable interrupts
            //
            am_hal_interrupt_master_set(ui32IntStatus);
            bIosInt = false;
            // Read & Clear the IOINT status
            iom_slave_read(bSpi, IOSOFFSET_INTSTAT, &data, 1);
            if ( data & HANDSHAKE_IOS_TO_IOM )
            {   // We need to clear the bit by writing to IOS
                data = HANDSHAKE_IOS_TO_IOM;
                iom_slave_write(bSpi, IOSOFFSET_INTCLR, &data, 1);
                iom_slave_read(bSpi, IOSOFFSET_HANDSHAKE, &ui8Resp, 1);
                if (AM_IOSTEST_CMD_ACK_DATA == ui8Resp)
                {
                    am_util_stdio_printf("\nHandshake success!\n");
                    g_pRefData[IOM_DATA_TAG] = ui8Tag;
                    g_pRefData[IOM_DATA_LENGTH] = ui8Length;
                    memcpy(&g_pRefData[IOM_DATA_VALUE], &g_pui8TestArray[ui32Offset], ui8Length - AM_IOS_HEADER_SIZE);
                    iom_slave_write(bSpi, 0, g_pRefData, ui8Length);
                    am_util_stdio_printf("\nStart data transmit!\n");
                    break;
                }
                else
                {   // Keep trying handshake
                    am_util_stdio_printf("\nHandshake fail, try again!\n");
                    data = AM_IOSTEST_CMD_START_DATA;
                    iom_slave_write(bSpi, IOSOFFSET_HANDSHAKE, &data, 1);
                }
            }
        }
        else
        {
            am_hal_sysctrl_sleep(AM_HAL_SYSCTRL_SLEEP_DEEP);
            //
            // Enable interrupts
            //
            am_hal_interrupt_master_set(ui32IntStatus);
        }
    }

    while (1)
    {
        //
        // Disable interrupt while we decide whether we're going to sleep.
        //
        uint32_t ui32IntStatus = am_hal_interrupt_master_disable();

        if ( bIosInt == true )
        {
            //
            // Enable interrupts
            //
            am_hal_interrupt_master_set(ui32IntStatus);
            bIosInt = false;
            // Read & Clear the IOINT status
            iom_slave_read(bSpi, IOSOFFSET_INTSTAT, &data, 1);
            if ( data & HANDSHAKE_IOS_TO_IOM )
            {   // We need to clear the bit by writing to IOS
                data = HANDSHAKE_IOS_TO_IOM;
                iom_slave_write(bSpi, IOSOFFSET_INTCLR, &data, 1);
                iom_slave_read(bSpi, 0, g_pIomRecvBuf, ui8Length);
                // Wait for read complete
                am_util_delay_us(100);
                // Verify received data at IOM
                ui8Resp = verify_result(ui8Length);
                if (ui8Resp)
                {
                    am_util_stdio_printf("\nTransmit success! Start address: 0x%x Data Size: 0x%x\n", ui32Offset, ui8Length - AM_IOS_HEADER_SIZE);
                    // Reset the buffers
                    for (i = 0; i < AM_IOS_LRAM_SIZE_MAX; i++)
                    {
                        g_pIomRecvBuf[i] = INBUFFER_EMPTY;
                        g_pRefData[i]    = INBUFFER_EMPTY;
                    }
                    ui32Offset += ui8Length - AM_IOS_HEADER_SIZE;
                    if (ui32Offset >= TEST_ARRAY_SIZE)
                    {
                        data = AM_IOSTEST_CMD_STOP_DATA;
                        iom_slave_write(bSpi, IOSOFFSET_HANDSHAKE, &data, 1);
                        am_util_stdio_printf("\nIOM transmit complete!\n");
                        break;
                    }
                    else
                    {
                        if (TEST_ARRAY_SIZE - ui32Offset >= AM_IOS_MAX_DATA_SIZE)
                        {
                            ui8Length = AM_IOS_LRAM_SIZE_MAX;
                        }
                        else
                        {
                            ui8Length = TEST_ARRAY_SIZE - ui32Offset + AM_IOS_HEADER_SIZE;
                        }
                        ui8Tag++;
                        g_pRefData[IOM_DATA_TAG] = ui8Tag;
                        g_pRefData[IOM_DATA_LENGTH] = ui8Length;
                        memcpy(&g_pRefData[IOM_DATA_VALUE], &g_pui8TestArray[ui32Offset], ui8Length - AM_IOS_HEADER_SIZE);
                        iom_slave_write(bSpi, 0, g_pRefData, ui8Length);
                    }
                }
                else
                {
                    am_util_stdio_printf("\nIOM transmit fail!\n");
                    break;
                }
            }
        }
        else
        {
            am_hal_sysctrl_sleep(AM_HAL_SYSCTRL_SLEEP_DEEP);
            //
            // Enable interrupts
            //
            am_hal_interrupt_master_set(ui32IntStatus);
        }
    }
    iom_clean_up(iom, bSpi);
    while (1)
    {
        //
        // Go to Deep Sleep.
        //
        am_hal_sysctrl_sleep(AM_HAL_SYSCTRL_SLEEP_DEEP);
    }
}


