//*****************************************************************************
//
//! @file am_hal_ble_patch.c
//!
//! @brief This is a binary patch for the BLE core.
//!
//! @addtogroup
//! @ingroup
//! @{
//
//*****************************************************************************

//*****************************************************************************
//
// Copyright (c) 2019, Ambiq Micro
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright notice,
// this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
// 
// 3. Neither the name of the copyright holder nor the names of its
// contributors may be used to endorse or promote products derived from this
// software without specific prior written permission.
// 
// Third party software included in this distribution is subject to the
// additional license terms as defined in the /docs/licenses directory.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.
//
// This is part of revision v2.2.0-7-g63f7c2ba1 of the AmbiqSuite Development Package.
//
//*****************************************************************************

#include "am_mcu_apollo.h"

//*****************************************************************************
//
// BLE LL local supported feature flags.
//
//     Bit position | Link Layer Feature
//     0            | LE Encryption
//     1            | Connection Parameters Request Procedure
//     2            | Extended Reject Indication
//     3            | Slave-initiated Features Exchange
//     4            | LE Ping
//     5            | LE Data Packet Length Extension
//     6            | LL Privacy
//     7            | Extended Scanner Filter Policies
//
// Specified 4.6 Feature Support, Link Layer Specification, Core V4.2.
//
//*****************************************************************************
#ifndef AM_HAL_BLE_LOCAL_FEATURE
#define AM_HAL_BLE_LOCAL_FEATURE    0x21
#endif

//*****************************************************************************
//
// Patches included in this file.
//
//*****************************************************************************
am_hal_ble_patch_t am_ble_buffer_patch;
am_hal_ble_patch_t am_ble_performance_patch;
am_hal_ble_patch_t am_ble_performance_copy_patch;
am_hal_ble_patch_t am_ble_nvds_patch;

//*****************************************************************************
//
// Patch application order.
//
//*****************************************************************************
am_hal_ble_patch_t *am_hal_ble_default_patch_array[] =
{
    // FTCODE patches (type 0xAA)

    // RAMCODE patches (type 0xBB)
    &am_ble_performance_patch,

    // Standard patches (type 0xCC)
    &am_ble_buffer_patch,

    // nvds param (type 0xDD)
    &am_ble_nvds_patch,
};

am_hal_ble_patch_t *am_hal_ble_default_copy_patch_array[] =
{
    // FTCODE patches (type 0xAA)

    // RAMCODE patches (type 0xBB)
    &am_ble_performance_copy_patch,

};

#define AM_HAL_BLE_NUM_DEFAULT_PATCHES                                        \
    (sizeof(am_hal_ble_default_patch_array) /                                 \
     sizeof(am_hal_ble_default_patch_array[0]))

am_hal_ble_patch_t **am_hal_ble_default_patches = am_hal_ble_default_patch_array;
am_hal_ble_patch_t **am_hal_ble_default_copy_patches = am_hal_ble_default_copy_patch_array;

const uint32_t am_hal_ble_num_default_patches = AM_HAL_BLE_NUM_DEFAULT_PATCHES;

//*****************************************************************************
//
// Patch Name:  RAMCODE COPY PATCH v1.10    for Apollo3 A1
//
// Bi-directional data fix
// Modulation deviation fix
// Extend patch memory
// Transmit speed patch
// Added AGC table and enabled AGC
// Added local feature support setting
// Fix to connection interval calculation issue with MTK chip sets (OPPO R15 fix)
// Set VCO to 250mv
// Modex auto calibration update
// Fix connection interval calculation issue
// Increase RF LDO ref voltage form 1.0v to 1.1v
// Decrease DAC channel delay cycle
// Increase the VCO swing from 250mv to 300mv
// Fix MD trans schedule issue (disabled)
// Fix link loss issue
// Reduce duration from TX to TX
// Optimized 32K XO frequency calculation
// Fix channel map rejected issue
// Optimized AGC Table
// Date:        2019-01-30
//*****************************************************************************

const am_hal_ble_buffer(0x0912) am_ble_performance_copy_patch_data =
{
    .bytes =
    {
        0x00,0x11,0x6e,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x01,0xc5,0x01,
        0x39,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,0x70,0xb5,0x00,0x20,0x0c,0x49,0x49,0x88,
        0x0c,0x4a,0x8b,0x18,0x1a,0x88,0x0c,0x49,0x9b,0x1c,0x00,0x24,0x13,0x25,0x2d,0x02,
        0x0c,0x54,0x40,0x1c,0xa8,0x42,0xfb,0xdb,0x00,0x20,0x00,0x2a,0x04,0xdd,0x1c,0x5c,
        0x0c,0x54,0x40,0x1c,0x90,0x42,0xfa,0xdb,0x04,0x48,0x80,0x47,0x00,0x20,0x70,0xbd,
        0x00,0x48,0x00,0x20,0x02,0x48,0x00,0x20,0x00,0x35,0x00,0x20,0xaf,0x33,0x01,0x00,
        0xa0,0x08,0x1f,0xb5,0x00,0x24,0x00,0x98,0x1d,0x28,0x43,0xd2,0x01,0x00,0x79,0x44,
        0x09,0x79,0x49,0x18,0x8f,0x44,0x0e,0x13,0x40,0x1a,0x25,0x40,0x40,0x40,0x40,0x40,
        0x40,0x40,0x40,0x40,0x40,0x40,0x2a,0x40,0x40,0x40,0x40,0x2d,0x40,0x32,0x40,0x35,
        0x38,0x40,0x40,0x00,0x01,0x98,0xc0,0xb2,0x00,0xf0,0x07,0xfa,0x2c,0xe0,0x02,0x98,
        0xc1,0xb2,0x01,0x98,0xc0,0xb2,0x00,0xf0,0xb8,0xf8,0x25,0xe0,0x06,0x98,0x83,0xb2,
        0x03,0x98,0x82,0xb2,0x02,0x98,0xc1,0xb2,0x01,0x98,0xc0,0xb2,0x00,0xf0,0x71,0xf9,
        0x1a,0xe0,0x01,0x98,0xc0,0xb2,0x00,0xf0,0x0a,0xf9,0x15,0xe0,0x00,0xf0,0x29,0xf8,
        0x12,0xe0,0x01,0x98,0x80,0xb2,0x00,0xf0,0x62,0xf9,0x0d,0xe0,0x00,0xf0,0x3b,0xfb,
        0x0a,0xe0,0x00,0xf0,0x94,0xfb,0x07,0xe0,0x01,0x98,0xc0,0xb2,0x00,0xf0,0xc7,0xfa,
        0x04,0x46,0x01,0xe0,0x00,0x24,0xe4,0x43,0x20,0x46,0x04,0xb0,0x10,0xbd,0x03,0xb4,
        0x01,0x48,0x01,0x90,0x01,0xbd,0x39,0x27,0x00,0x00,0x03,0xb4,0x01,0x48,0x01,0x90,
        0x01,0xbd,0x95,0x28,0x00,0x00,0x03,0xb4,0x01,0x48,0x01,0x90,0x01,0xbd,0x01,0x01,
        0x00,0x00,0xf0,0xb4,0x00,0x20,0x43,0x22,0x12,0x06,0x51,0x68,0xff,0x24,0x01,0x34,
        0x21,0x43,0x51,0x60,0x51,0x68,0x23,0x03,0x19,0x43,0x51,0x60,0xa3,0x23,0xdb,0x05,
        0x19,0x68,0x49,0x08,0x49,0x00,0x19,0x60,0x2a,0x49,0x09,0x69,0xce,0xb2,0x29,0x4d,
        0x2a,0x4f,0x29,0x88,0xb9,0x42,0x01,0xd3,0x04,0x20,0x0d,0xe0,0x28,0x4f,0xb9,0x42,
        0x01,0xd3,0x03,0x20,0x08,0xe0,0x26,0x4f,0xb9,0x42,0x01,0xd3,0x02,0x20,0x03,0xe0,
        0x25,0x4f,0xb9,0x42,0x00,0xd3,0x01,0x20,0x24,0x4f,0x39,0x18,0x20,0x31,0x09,0x7e,
        0xb1,0x42,0x09,0xda,0x00,0x28,0x01,0xdd,0x40,0x1e,0x40,0xb2,0x39,0x18,0x09,0x7a,
        0x40,0x00,0xc0,0x19,0x00,0x8b,0x0b,0xe0,0x04,0x28,0x04,0xda,0x39,0x5c,0xb1,0x42,
        0x01,0xdb,0x40,0x1c,0x40,0xb2,0x39,0x18,0x09,0x7a,0x40,0x00,0xc0,0x19,0x00,0x8b,
        0x17,0x4e,0x31,0x62,0x19,0x68,0x49,0x08,0x49,0x00,0x19,0x60,0x15,0x4e,0x31,0x6b,
        0x0f,0x46,0x27,0x43,0x37,0x63,0x98,0x62,0xa1,0x43,0x31,0x63,0x28,0x80,0x51,0x68,
        0xb0,0x03,0x81,0x43,0x10,0x48,0x00,0x78,0xc0,0x07,0xc0,0x0f,0x03,0x05,0x19,0x43,
        0x51,0x60,0x51,0x68,0x00,0x02,0xa1,0x43,0x01,0x43,0x51,0x60,0xf0,0xbc,0x70,0x47,
        0x00,0x00,0x80,0x00,0x80,0x45,0x4e,0x60,0x00,0x20,0xf6,0x3f,0x00,0x00,0xf6,0x38,
        0x00,0x00,0xf6,0x2d,0x00,0x00,0xf6,0x09,0x00,0x00,0xfc,0x67,0x00,0x20,0x80,0x04,
        0xc0,0x50,0x40,0x00,0x80,0x45,0x3a,0x60,0x00,0x20,0xf0,0xb5,0x82,0xb0,0x43,0x22,
        0x12,0x06,0x53,0x68,0x01,0x24,0x64,0x04,0x23,0x43,0x53,0x60,0xca,0x07,0xd2,0x0f,
        0x96,0x46,0x8a,0x07,0xd3,0x0f,0x4a,0x07,0xd4,0x0f,0x0a,0x07,0xd2,0x0f,0x01,0x92,
        0xca,0x06,0xd2,0x0f,0x00,0x92,0x8a,0x06,0xd2,0x0f,0x94,0x46,0x4a,0x06,0xd5,0x0f,
        0xce,0x09,0x1f,0x4a,0x11,0x68,0x03,0x27,0x7f,0x05,0xb9,0x43,0x12,0x69,0x1d,0x4f,
        0x3a,0x40,0x00,0x28,0x06,0xd0,0x01,0x28,0x09,0xd0,0x01,0x27,0xbf,0x05,0x02,0x28,
        0x03,0xd0,0x39,0x43,0x00,0x28,0x06,0xd0,0x1b,0xe0,0x39,0x43,0xfa,0xe7,0x01,0x27,
        0x7f,0x05,0x39,0x43,0xf6,0xe7,0x30,0x03,0x10,0x43,0x6a,0x01,0x10,0x43,0x62,0x46,
        0xd2,0x02,0x10,0x43,0x00,0x9a,0xd2,0x01,0x10,0x43,0x01,0x9a,0x92,0x01,0x10,0x43,
        0xa2,0x02,0x10,0x43,0x5a,0x02,0x10,0x43,0x72,0x46,0x12,0x02,0x10,0x43,0x08,0x4a,
        0x10,0x61,0x07,0x48,0x01,0x60,0x43,0x20,0x00,0x06,0x41,0x68,0x01,0x22,0x52,0x04,
        0x91,0x43,0x05,0x4a,0x12,0x78,0xd2,0x07,0x92,0x0b,0x11,0x43,0x41,0x60,0x02,0xb0,
        0xf0,0xbd,0x00,0x00,0xc0,0x43,0x03,0xe0,0xff,0xff,0x3a,0x60,0x00,0x20,0xf8,0xb5,
        0x00,0x24,0x43,0x20,0x00,0x06,0x41,0x68,0x01,0x27,0xbf,0x03,0x39,0x43,0x41,0x60,
        0x41,0x68,0xba,0x00,0x11,0x43,0x41,0x60,0x21,0x48,0x05,0x68,0x51,0x1c,0x0d,0x43,
        0x05,0x60,0x1f,0x4e,0xc8,0x20,0xb0,0x47,0x1f,0x49,0x88,0x68,0x38,0x43,0x88,0x60,
        0x0e,0x46,0xf0,0x68,0x00,0x04,0xc7,0x0f,0x1a,0x49,0x01,0x20,0x88,0x47,0x20,0x46,
        0x1a,0x49,0x64,0x1c,0x88,0x42,0x01,0xd8,0x00,0x2f,0xf2,0xd0,0x18,0x48,0x05,0x40,
        0x13,0x48,0x05,0x60,0x01,0x20,0x40,0x03,0xb0,0x60,0xf1,0x68,0x15,0x48,0x01,0x40,
        0x70,0x68,0x14,0x4a,0x10,0x40,0x08,0x43,0x70,0x60,0x30,0x68,0x3f,0x21,0x89,0x02,
        0x88,0x43,0x30,0x60,0x43,0x21,0x09,0x06,0x4a,0x68,0x01,0x20,0x80,0x03,0x82,0x43,
        0x0e,0x48,0x00,0x78,0xc0,0x07,0xc0,0x0f,0x83,0x03,0x1a,0x43,0x4a,0x60,0x4a,0x68,
        0x01,0x23,0x1b,0x04,0x9a,0x43,0x00,0x04,0x02,0x43,0x4a,0x60,0xf8,0xbd,0x00,0x00,
        0x40,0x52,0xa1,0x3b,0x00,0x00,0x00,0x00,0xc0,0x51,0x10,0x27,0x00,0x00,0xfe,0xff,
        0xfe,0xff,0x3f,0x3f,0x00,0x00,0xc0,0xc0,0xff,0xff,0x3a,0x60,0x00,0x20,0x70,0x47,
        0x00,0x00,0xf0,0xb5,0x83,0xb0,0x08,0x25,0x3c,0x49,0x00,0x20,0x08,0x70,0x43,0x20,
        0x00,0x06,0x41,0x68,0xaa,0x02,0x11,0x43,0x41,0x60,0x29,0x27,0x38,0x49,0x7f,0x06,
        0xf9,0x60,0x41,0x68,0x37,0x4b,0x91,0x43,0x1b,0x78,0xdb,0x07,0x9b,0x0c,0x19,0x43,
        0x41,0x60,0x25,0x24,0xa4,0x01,0x41,0x68,0x11,0x43,0x41,0x60,0x41,0x68,0x52,0x10,
        0x11,0x43,0x41,0x60,0xf8,0x68,0x03,0x21,0x49,0x06,0x88,0x43,0x56,0x03,0x06,0x43,
        0x90,0x03,0x06,0x43,0xb8,0x68,0x01,0x20,0xff,0xf7,0x81,0xfe,0x21,0x46,0x10,0x20,
        0xff,0xf7,0x83,0xfe,0x28,0x03,0x06,0x43,0x00,0x24,0x00,0x2d,0x0c,0xd9,0xfe,0x60,
        0x25,0x49,0x64,0x20,0x88,0x47,0xf8,0x68,0x69,0x46,0x00,0x0a,0x80,0x1c,0x08,0x55,
        0x64,0x1c,0xe4,0xb2,0xac,0x42,0xf2,0xd3,0x6a,0x46,0xd0,0x79,0x91,0x79,0x40,0x18,
        0x51,0x79,0x12,0x79,0x89,0x18,0x40,0x18,0x6a,0x46,0xd1,0x78,0x41,0x18,0x90,0x78,
        0x09,0x18,0x50,0x78,0x09,0x18,0x10,0x78,0x08,0x18,0x29,0x46,0xff,0xf7,0x63,0xfe,
        0x12,0x49,0xc0,0xb2,0x08,0x70,0x39,0x68,0x01,0x22,0x92,0x02,0x11,0x43,0x39,0x60,
        0x40,0x00,0x40,0x1c,0x78,0x61,0x00,0x20,0xff,0xf7,0x49,0xfe,0x43,0x21,0x09,0x06,
        0x4a,0x68,0x01,0x20,0x00,0x03,0x82,0x43,0x0a,0x48,0x00,0x78,0xc0,0x07,0xc0,0x0f,
        0x03,0x03,0x1a,0x43,0x4a,0x60,0x4a,0x68,0x01,0x23,0x5b,0x03,0x9a,0x43,0x40,0x03,
        0x02,0x43,0x4a,0x60,0x03,0xb0,0xf0,0xbd,0x00,0x00,0x70,0x60,0x00,0x20,0xcc,0x34,
        0x63,0x02,0x3a,0x60,0x00,0x20,0xa1,0x3b,0x00,0x00,0x70,0xb4,0x43,0x21,0x09,0x06,
        0x48,0x68,0x01,0x24,0xa4,0x04,0x20,0x43,0x48,0x60,0xc4,0x20,0x87,0x22,0xd2,0x05,
        0x10,0x60,0x5c,0x48,0x50,0x61,0x48,0x68,0x5c,0x4a,0xa0,0x43,0x12,0x78,0xd2,0x07,
        0xd2,0x0f,0x93,0x04,0x18,0x43,0x48,0x60,0x8b,0x20,0x58,0x4b,0xc0,0x05,0x43,0x63,
        0x58,0x4b,0x01,0x25,0xdd,0x60,0x06,0x25,0xcd,0x60,0x05,0x25,0xc5,0x63,0x85,0x68,
        0x6d,0x08,0xf0,0x3d,0x85,0x60,0xc5,0x68,0x6d,0x08,0x5d,0x35,0xc5,0x60,0x05,0x69,
        0x6d,0x08,0x05,0x61,0x45,0x69,0x6d,0x08,0x45,0x61,0x85,0x69,0x6d,0x08,0x85,0x61,
        0xc5,0x69,0x6d,0x08,0xc5,0x61,0x05,0x6a,0x6d,0x08,0x05,0x62,0x45,0x6a,0x6d,0x08,
        0x45,0x62,0x85,0x6a,0x6d,0x08,0x85,0x62,0xc5,0x6a,0x6d,0x08,0xc5,0x62,0x01,0x25,
        0x1d,0x61,0x5d,0x62,0x9d,0x63,0x43,0x4b,0x1d,0x60,0x43,0x4d,0x5d,0x61,0x1d,0x6a,
        0x6d,0x08,0x1d,0x62,0xc3,0x6a,0x01,0x25,0x2b,0x43,0xc3,0x62,0x48,0x68,0xa3,0x10,
        0x18,0x43,0x48,0x60,0x48,0x68,0x9b,0x10,0x18,0x43,0x48,0x60,0x3d,0x48,0x3b,0x4b,
        0x43,0x61,0x83,0x68,0x3f,0x25,0xad,0x05,0x2b,0x43,0x83,0x60,0x00,0x23,0xc3,0x60,
        0x39,0x4b,0x83,0x61,0x39,0x4d,0x2b,0x68,0x1e,0x26,0xb3,0x43,0x2b,0x60,0x83,0x6a,
        0xf5,0x03,0xab,0x43,0x1b,0x19,0x83,0x62,0x48,0x68,0xa3,0x10,0x98,0x43,0x13,0x04,
        0x18,0x43,0x48,0x60,0x48,0x68,0x23,0x11,0x98,0x43,0x93,0x03,0x18,0x43,0x48,0x60,
        0x48,0x68,0x63,0x11,0x18,0x43,0x48,0x60,0x29,0x20,0x40,0x06,0x04,0x6a,0x24,0x09,
        0x24,0x01,0x08,0x34,0x04,0x62,0x48,0x68,0x98,0x43,0x53,0x03,0x18,0x43,0x48,0x60,
        0x48,0x68,0x01,0x23,0x1b,0x05,0x18,0x43,0x48,0x60,0x28,0x24,0xa3,0x20,0xc0,0x05,
        0x04,0x60,0x22,0x4d,0x2d,0x88,0x85,0x62,0x48,0x68,0x12,0x05,0x98,0x43,0x10,0x43,
        0x48,0x60,0x1f,0x48,0xe6,0x21,0x01,0x70,0x04,0x72,0x1d,0x4a,0x1e,0x48,0x10,0x83,
        0x1e,0x48,0xe0,0x23,0x03,0x76,0x1a,0x4c,0xd4,0x22,0x62,0x70,0x3c,0x22,0x62,0x72,
        0x1b,0x4a,0x62,0x83,0x41,0x76,0xc8,0x22,0xa2,0x70,0x15,0x4d,0x46,0x24,0xac,0x72,
        0x18,0x4c,0xac,0x83,0x81,0x76,0x29,0x46,0xca,0x70,0x50,0x21,0x2a,0x46,0xd1,0x72,
        0x15,0x49,0xd1,0x83,0xc3,0x76,0x5a,0x21,0x11,0x73,0x13,0x49,0x11,0x84,0x03,0x77,
        0x70,0xbc,0x70,0x47,0x00,0x00,0xff,0x7f,0x00,0x00,0x3a,0x60,0x00,0x20,0x49,0x02,
        0x00,0x00,0x40,0x00,0x80,0x45,0x80,0x00,0x80,0x45,0x1e,0x02,0x00,0x00,0x03,0x00,
        0x3c,0x00,0x00,0x00,0x40,0x52,0x08,0x00,0x0f,0x00,0x00,0x00,0xc0,0x51,0x4e,0x60,
        0x00,0x20,0xfc,0x67,0x00,0x20,0xf6,0x07,0x00,0x00,0x1c,0x68,0x00,0x20,0xf6,0x09,
        0x00,0x00,0xf6,0x2d,0x00,0x00,0xf6,0x38,0x00,0x00,0xf6,0x3f,0x00,0x00,0xf8,0xb5,
        0x2d,0x48,0x00,0x68,0x00,0x28,0x54,0xd1,0x43,0x22,0x12,0x06,0x50,0x68,0x01,0x21,
        0xc9,0x03,0x08,0x43,0x50,0x60,0x28,0x4f,0x3c,0x68,0x01,0x25,0x03,0x20,0x00,0x06,
        0x20,0x43,0x38,0x60,0x00,0x26,0x25,0x49,0x1e,0x20,0x88,0x47,0x24,0x49,0x01,0x20,
        0x88,0x47,0x78,0x68,0xc0,0x07,0xc0,0x0f,0x31,0x46,0x21,0x4a,0x76,0x1c,0x91,0x42,
        0x01,0xd8,0x00,0x28,0xf2,0xd0,0x1c,0x48,0x81,0x68,0x1e,0x48,0x01,0x60,0x00,0x20,
        0x00,0x26,0x00,0x2d,0x11,0xd0,0x1c,0x4b,0x32,0x46,0x00,0x25,0xcb,0x1a,0xaa,0x41,
        0x14,0xda,0x40,0x1c,0x05,0x46,0x18,0x4f,0x4d,0x43,0x33,0x46,0x00,0x22,0x7d,0x1b,
        0x9a,0x41,0x4d,0x1b,0x93,0x41,0xf4,0xdb,0x08,0xe0,0x14,0x4b,0x99,0x42,0x05,0xd2,
        0x40,0x1c,0x02,0x46,0x4a,0x43,0x9a,0x1a,0x8a,0x42,0xf9,0xd8,0x01,0x21,0x09,0x06,
        0x8c,0x43,0x49,0x00,0x0c,0x43,0x08,0x49,0x0c,0x60,0x43,0x22,0x12,0x06,0x51,0x68,
        0x01,0x23,0xdb,0x03,0x99,0x43,0x0a,0x4b,0x1b,0x78,0xdb,0x07,0x1b,0x0c,0x19,0x43,
        0x51,0x60,0xf8,0xbd,0x00,0x00,0x5c,0x60,0x00,0x20,0x00,0x00,0x40,0x44,0xa1,0x3b,
        0x00,0x00,0x10,0x27,0x00,0x00,0x60,0x60,0x00,0x20,0x00,0x20,0xbc,0xbe,0x00,0xd0,
        0x12,0x13,0x3a,0x60,0x00,0x20,0x70,0xb5,0x23,0x48,0x80,0x47,0x43,0x24,0x24,0x06,
        0x01,0x21,0x60,0x68,0x89,0x04,0x08,0x43,0x60,0x60,0x1f,0x48,0x80,0x47,0x1f,0x48,
        0x00,0x78,0xaa,0x28,0x06,0xd1,0xa0,0x68,0x80,0x07,0x03,0xd1,0x1d,0x49,0x88,0x47,
        0x1d,0x49,0x08,0x60,0xa1,0x20,0xc0,0x05,0x02,0x6b,0x02,0x21,0x8a,0x43,0x0a,0x43,
        0x02,0x63,0x01,0x21,0x02,0x6b,0x8a,0x43,0x0a,0x43,0x02,0x63,0x01,0x6b,0x04,0x26,
        0xb1,0x43,0x31,0x43,0x01,0x63,0x14,0x4c,0x60,0x68,0x14,0x4d,0x80,0x00,0x04,0xd5,
        0x01,0x20,0xa8,0x47,0x60,0x68,0x80,0x00,0xfa,0xd4,0x11,0x4c,0x0f,0x20,0x60,0x60,
        0x01,0x20,0xa8,0x47,0x07,0x20,0x60,0x60,0x04,0x20,0xa8,0x47,0x06,0x20,0x60,0x60,
        0x8b,0x21,0x0c,0x48,0xc9,0x05,0x08,0x60,0x0c,0x48,0x01,0x69,0x31,0x43,0x01,0x61,
        0x30,0xbf,0x70,0xbd,0x00,0x00,0x35,0x37,0x00,0x00,0x39,0x9c,0x00,0x00,0x2c,0x60,
        0x00,0x20,0xd1,0x39,0x00,0x00,0x54,0x60,0x00,0x20,0x80,0x00,0x80,0x45,0xa1,0x3b,
        0x00,0x00,0x40,0x00,0x80,0x45,0x26,0x03,0x00,0x00,0x00,0xed,0x00,0xe0,0x70,0xb5,
        0x2f,0x4d,0x0f,0x20,0x68,0x60,0x8b,0x24,0xe4,0x05,0x20,0x68,0x01,0x21,0x49,0x02,
        0x88,0x43,0x20,0x60,0x2b,0x48,0x80,0x47,0x01,0x20,0x80,0xf3,0x10,0x88,0x29,0x48,
        0x40,0x68,0x29,0x49,0x80,0x00,0x06,0xd4,0x20,0x68,0x08,0x22,0x10,0x43,0x20,0x60,
        0x01,0x20,0x88,0x47,0x01,0xe0,0x01,0x20,0x88,0x47,0x24,0x48,0x80,0x47,0x24,0x48,
        0x25,0x49,0x00,0x78,0x88,0x47,0x0d,0x20,0x68,0x60,0x23,0x48,0x80,0x47,0x43,0x20,
        0x00,0x06,0x41,0x68,0x01,0x23,0x5b,0x03,0x19,0x43,0x41,0x60,0x29,0x21,0x1f,0x4a,
        0x49,0x06,0xca,0x60,0x0a,0x6a,0x12,0x09,0x12,0x01,0x08,0x32,0x0a,0x62,0x0a,0x68,
        0xdc,0x10,0x22,0x43,0x0a,0x60,0x1a,0x4a,0x12,0x78,0x52,0x00,0x52,0x1c,0x4a,0x61,
        0x42,0x68,0x18,0x49,0x9a,0x43,0x09,0x78,0xc9,0x07,0xc9,0x0f,0x4b,0x03,0x1a,0x43,
        0x42,0x60,0x43,0x68,0xa2,0x02,0x13,0x43,0x43,0x60,0x28,0x24,0xa3,0x23,0xdb,0x05,
        0x1c,0x60,0x11,0x4c,0x24,0x88,0x9c,0x62,0x43,0x68,0x09,0x05,0x93,0x43,0x0b,0x43,
        0x43,0x60,0x00,0x20,0x80,0xf3,0x10,0x88,0x0d,0x48,0x80,0x47,0x70,0xbd,0x40,0x00,
        0x80,0x45,0x85,0x3b,0x01,0x00,0x80,0x00,0x80,0x45,0x89,0x44,0x01,0x00,0x6d,0x34,
        0x01,0x00,0x40,0x60,0x00,0x20,0xa1,0x3b,0x00,0x00,0x99,0x4d,0x01,0x00,0xcc,0x34,
        0x63,0x04,0x70,0x60,0x00,0x20,0x3a,0x60,0x00,0x20,0x4e,0x60,0x00,0x20,0xad,0x34,
        0x01,0x00
    }
};

am_hal_ble_patch_t am_ble_performance_copy_patch =
{
    .ui32Type = 0xBB,
    .ui32Length = 0x0912,
    .ui32CRC = 0x9516,
    .pui32Data = am_ble_performance_copy_patch_data.words,
};

//*****************************************************************************
//
// Patch Name:  RAMCODE PATCH v1.10    for Apollo3 A1
//
// Bi-directional data fix
// Modulation deviation fix
// Extend patch memory
// Transmit speed patch
// Added AGC table and enabled AGC
// Added local feature support setting
// Fix to connection interval calculation issue with MTK chip sets (OPPO R15 fix)
// Set VCO to 250mv
// Modex auto calibration update
// Fix connection interval calculation issue
// Increase RF LDO ref voltage form 1.0v to 1.1v
// Decrease DAC channel delay cycle
// Increase the VCO swing from 250mv to 300mv
// Fix MD trans schedule issue (disabled)
// Fix link loss issue
// Reduce duration from TX to TX
// Optimized 32K XO frequency calculation
// Fix channel map rejected issue
// Optimized AGC Table
// Date:        2019-01-30
//*****************************************************************************

am_hal_ble_buffer(0x0104) am_ble_performance_patch_data =
{
    .bytes =
    {
        0x00,0x11,0x02,0x01,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x01,0xc5,0x01,
        0x39,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x1b,0x00,0xa1,0x06,0x1f,0xb5,0x82,0xb0,0x08,0x98,0x00,0x90,
        0x02,0xa8,0x0f,0xc8,0x01,0x4c,0xa0,0x47,0x06,0xb0,0x10,0xbd,0x01,0x35,0x00,0x20,
        0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x08,0x48,0x80,0x47,0x00,0xbf,0x00,0xbf,
        0x02,0x2d,0x05,0xd1,0x06,0x48,0x80,0x47,0x00,0xbf,0x00,0xbf,0x05,0x48,0x80,0x47,
        0x00,0x21,0x03,0x9a,0x04,0x98,0x90,0x47,0x03,0x48,0x00,0x47,0x99,0x4a,0x01,0x00,
        0x25,0x4b,0x01,0x00,0xaf,0x4a,0x01,0x00,0x8f,0x4c,0x01,0x00,0x00,0x00,0x00,0x00,
        0x04,0x48,0x01,0x68,0x28,0x22,0x11,0x43,0x50,0x22,0x91,0x43,0x01,0x60,0x00,0xbf,
        0x01,0x48,0x00,0x47,0x00,0x00,0xc0,0x52,0x63,0x2a,0x00,0x00,0x00,0x00,0x00,0x00,
        0x04,0x48,0x01,0x68,0x50,0x22,0x11,0x43,0x28,0x22,0x91,0x43,0x01,0x60,0x00,0xbf,
        0x01,0x48,0x00,0x47,0x00,0x00,0xc0,0x52,0x83,0x2a,0x00,0x00,0x00,0xbf,0x00,0xbf,
        0x00,0xbf,0x00,0xbf,0x08,0x98,0x00,0x28,0x01,0xd0,0x01,0x20,0x02,0x90,0x00,0x20,
        0x60,0x85,0x01,0x48,0x00,0x47,0x00,0xbf,0xd5,0xed,0x00,0x00,0x00,0xbf,0x00,0xbf,
        0x60,0x88,0x00,0x28,0x04,0xd1,0x10,0x7d,0x08,0x28,0x01,0xd3,0x04,0x20,0x10,0x75,
        0x02,0x98,0x81,0x79,0x01,0x20,0x01,0x43,0x02,0x98,0x81,0x71,0x00,0x48,0x00,0x47,
        0xa5,0xf7,0x00,0x00
    }
};

am_hal_ble_patch_t am_ble_performance_patch =
{
    .ui32Type = 0xBB,
    .ui32Length = 0x0104,
    .ui32CRC = 0x933d,
    .pui32Data = am_ble_performance_patch_data.words,
};

//*****************************************************************************
//
// Patch Name:  Function PATCH v1.10    for Apollo3 A1
//
// Bi-directional data fix
// Modulation deviation fix
// Extend patch memory
// Transmit speed patch
// Added AGC table and enabled AGC
// Added local feature support setting
// Fix to connection interval calculation issue with MTK chip sets (OPPO R15 fix)
// Set VCO to 250mv
// Modex auto calibration update
// Fix connection interval calculation issue
// Increase RF LDO ref voltage form 1.0v to 1.1v
// Decrease DAC channel delay cycle
// Increase the VCO swing from 250mv to 300mv
// Fix MD trans schedule issue (disabled)
// Fix link loss issue
// Reduce duration from TX to TX
// Optimized 32K XO frequency calculation
// Fix channel map rejected issue
// Optimized AGC Table
// Date:        2019-01-30
//*****************************************************************************

const am_hal_ble_buffer(0x0d38) am_ble_buffer_patch_data =
{
    .bytes =
    {
        0x00,0x22,0x38,0x0d,0xff,0xff,0x00,0x00,0x32,0x35,0x09,0x00,0x65,0x39,0x09,0x00,
        0x2b,0x45,0x09,0x00,0xa9,0x48,0x09,0x00,0xf7,0x53,0x09,0x00,0x1a,0x5c,0x09,0x00,
        0x1c,0x64,0x09,0x00,0xfd,0x6a,0x09,0x00,0x1a,0x75,0x09,0x00,0xde,0x7b,0x09,0x00,
        0x4b,0x85,0x09,0x00,0xb3,0x8b,0x09,0x00,0x1f,0x95,0x09,0x00,0x4f,0x9c,0x09,0x00,
        0xf5,0xa2,0x09,0x00,0x1e,0xad,0x09,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x47,
        0x41,0x48,0x00,0x20,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x05,0xb0,0xf0,0xbd,0x00,0x00,
        0x90,0x67,0x00,0x20,0x10,0x27,0x00,0x00,0x00,0x10,0x00,0x20,0x88,0x13,0x00,0x00,
        0x18,0x10,0x00,0x20,0xff,0x03,0x00,0x00,0xff,0xb5,0xff,0xb0,0x82,0xb0,0x07,0x46,
        0x0c,0x46,0x16,0x46,0x00,0x25,0x30,0x48,0x06,0x60,0x84,0x99,0x04,0xd0,0xee,0x28,
        0x02,0xd0,0x03,0x20,0x60,0x73,0x70,0xbd,0x60,0x7c,0xf1,0x28,0x06,0xd0,0xf2,0x28,
        0x04,0xd0,0xf3,0x28,0x02,0xd0,0x02,0x20,0x60,0x73,0x70,0xbd,0x00,0x20,0x60,0x73,
        0x70,0xbd,0x00,0x00,0x18,0x10,0x00,0x20,0x0a,0x10,0x00,0x20,0x00,0x48,0x00,0x47,
        0x81,0x4d,0x00,0x20,0x70,0x88,0x00,0x28,0x16,0xd1,0x14,0x20,0x01,0x21,0x0b,0x20,
        0xed,0xf7,0xc6,0xfc,0x10,0xbd,0x00,0x00,0x00,0x48,0x00,0x47,0x15,0x4e,0x00,0x20,
        0x00,0x28,0x02,0xd0,0x08,0x78,0x01,0x28,0x1a,0xd0,0x08,0x78,0x02,0x28,0x17,0xd0,
        0x00,0x28,0x0e,0xd0,0x01,0x28,0x0c,0xd0,0xf0,0xf7,0xd9,0xfe,0x00,0x28,0x08,0xd0,
        0x00,0xf0,0x16,0xf8,0x32,0x20,0xef,0xf7,0x51,0xf8,0xf0,0xf7,0x80,0xe1,0x00,0xe0,
        0x00,0xe1,0x00,0xe0,0x02,0x48,0x01,0x68,0x28,0x22,0x91,0x43,0x01,0x60,0x70,0x47,
        0x00,0x00,0xc0,0x52,0x00,0x48,0x00,0x47,0x81,0x48,0x00,0x20,0x01,0x60,0x70,0x47,
        0x00,0x00,0xc0,0x52,0x02,0x48,0x01,0x68,0x50,0x22,0x91,0x43,0x01,0x60,0x70,0x47,
        0x00,0x00,0xc0,0x52,0x00,0x48,0x00,0x47,0xa1,0x48,0x00,0x20,0xc0,0x40,0x80,0x50,
        0x10,0xb5,0x0b,0x46,0x11,0x46,0x02,0x24,0x0c,0x22,0x50,0x43,0x06,0x4a,0x80,0x18,
        0x00,0x28,0x06,0xd0,0x42,0x68,0x00,0x2a,0x03,0xd0,0x18,0x46,0x00,0xf0,0x10,0xfb,
        0x04,0x46,0x20,0x46,0x10,0xbd,0x00,0x00,0x98,0x56,0x01,0x00,0x00,0x49,0x08,0x47,
        0x99,0x4e,0x00,0x20,0xf3,0xf7,0x5c,0xfd,0x7c,0x20,0x00,0x5b,0x0b,0xf8,0x40,0x19,
        0xc1,0xb2,0x00,0x29,0x03,0xd0,0x40,0x34,0xa0,0x8f,0xf9,0xf7,0x73,0xfd,0x70,0xbd,
        0x01,0x21,0xe1,0xe7,0x00,0x49,0x08,0x47,0x81,0x50,0x00,0x20,0x80,0xf3,0x10,0x88,
        0x28,0x46,0xf3,0xf7,0x4d,0xf9,0x00,0x21,0x81,0xf3,0x10,0x88,0x00,0x28,0x0f,0xd0,
        0x81,0x88,0x0a,0x46,0x0a,0x3a,0x46,0x2a,0x0c,0xd2,0x64,0x1c,0xd2,0x0b,0xd2,0x03,
        0x0a,0x43,0xc2,0x84,0x00,0x20,0x80,0xf3,0x10,0x88,0x70,0x47,0x66,0x04,0x00,0x00,
        0x40,0x44,0x80,0x50,0x00,0x49,0x08,0x47,0xed,0x50,0x00,0x20,0x11,0x90,0x80,0x8f,
        0x0e,0x90,0x12,0x98,0x08,0x30,0x0d,0x90,0x12,0x98,0x30,0x30,0x0c,0x90,0x12,0x98,
        0x44,0x30,0x0b,0x90,0x00,0x20,0x0a,0x90,0x01,0x25,0x0c,0x98,0x07,0xf0,0xa4,0xf8,
        0x00,0x28,0x2d,0xd0,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,
        0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x91,0xe0,0x0c,0x9a,0xff,0x20,0x22,0x23,
        0x11,0x46,0x0a,0x30,0xfd,0xf7,0x9e,0xfb,0x07,0x70,0x61,0x88,0xc1,0x81,0xa1,0x88,
        0x01,0x82,0xe1,0x88,0xc1,0x80,0x21,0x89,0x01,0x81,0xa1,0x7a,0xc0,0x01,0x23,0xe0,
        0x77,0xe0,0x02,0x98,0x08,0x1a,0x40,0x01,0x40,0x09,0x90,0x42,0x1c,0xd9,0x02,0x98,
        0x40,0x1a,0x40,0x01,0x40,0x09,0x40,0x42,0x16,0xe0,0x02,0x98,0x00,0xbf,0x00,0xbf,
        0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0xbf,0x00,0x20,
        0x0a,0xe0,0x02,0x98,0x38,0x1a,0x40,0x01,0x40,0x09,0x90,0x42,0xc1,0x65,0x75,0x60,
        0x60,0x89,0xb8,0x84,0x04,0x9a,0x01,0x20,0x50,0x75,0x60,0x88,0xf8,0x85,0x04,0x9a,
        0x60,0x78,0x10,0x75,0x00,0x48,0x00,0x47,0xe1,0x48,0x00,0x20,0x00,0xbf,0x00,0x00,
        0x60,0x89,0x3a,0x8d,0x40,0x1e,0x80,0xb2,0x82,0x42,0x03,0xd1,0x08,0x20,0x01,0x43,
        0x03,0x98,0x81,0x71,0x07,0xb0,0xf0,0xbd,0x60,0x61,0x00,0x20,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        AM_HAL_BLE_LOCAL_FEATURE,//0x01,
        0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,
        0xff,0x03,0x00,0x00,0xff,0x00,0x3c,0x1f,0x00,0x00,0x00,0x00,0x01,0x20,0x00,0x00,
        0x89,0x7d,0x00,0x00,0x02,0x20,0x00,0x00,0xfd,0x76,0x00,0x00,0x10,0x18,0x00,0x8c,
        0xc0,0x0b,0x08,0x9b,0x5b,0x1e,0x18,0x42,0x05,0xd1,0x08,0x98,0x00,0x48,0x00,0x47,
        0xc1,0x48,0x00,0x20,0x00,0xbf,0x00,0x00,0x80,0x4b,0xd2,0x18,0x50,0x84,0x01,0x20,
        0x80,0xf3,0x10,0x88,0xb8,0x68,0x00,0x28,0x16,0xd0,0x00,0x20,0x00,0x28,0x19,0xd0,
        0x38,0x6b,0x00,0x28,0x12,0xd0,0x00,0x20,0x00,0x28,0x13,0xd0,0x9f,0xfc,0x00,0x19,
        0x06,0x4b,0x59,0x68,0x09,0x18,0x8a,0x08,0x94,0x00,0x09,0x1b,0x59,0x60,0x41,0x01,
        0x08,0x1a,0x80,0x18,0x10,0xbd,0x00,0x00,0x2c,0x60,0x00,0x20,0xb8,0x67,0x00,0x20,
        0x00,0x49,0x08,0x47,0x79,0x55,0x00,0x20,0x7d,0x21,0x80,0x6a,0x09,0x02,0x88,0x42,
        0x02,0xd3,0x40,0x1a,0x01,0x22,0x00,0xe0,0x08,0x1a,0x00,0x2a,0x29,0x19,0x08,0x1a,
        0x21,0x46,0xee,0xf7,0xb1,0xfe,0x88,0xb2,0x70,0xbd,0x28,0x30,0x60,0x30,0x00,0x7e,
        0x70,0x47,0x10,0xb4,0x02,0x46,0x40,0x32,0xd3,0x8c,0x06,0x24,0x63,0x43,0x14,0x8d,
        0xe3,0x18,0x9b,0xb2,0x93,0x84,0x60,0x30,0x41,0x75,0x18,0x46,0x10,0xbc,0x70,0x47,
        0x28,0x30,0x60,0x30,0x40,0x7d,0x00,0x28,0x00,0x20,0x00,0x20,0x40,0x79,0x00,0x07,
        0x02,0xd5,0x04,0x20,0x60,0x70,0x34,0xe0,0x28,0x69,0x05,0xf0,0x51,0xfb,0x00,0x28,
        0x02,0xd0,0x02,0x20,0x60,0x70,0x2c,0xe0,0x03,0x98,0x03,0xf0,0x4d,0xfe,0x04,0x28,
        0x07,0xd1,0x68,0x46,0x01,0x79,0x02,0x20,0x88,0x43,0x05,0xd0,0x00,0x20,0x00,0x28,
        0x04,0xd0,0x00,0x20,0x60,0x70,0x1c,0xe0,0x01,0x20,0xf8,0xe7,0x58,0x67,0x00,0x20,
        0x40,0x00,0x80,0x50,0xb8,0x67,0x00,0x20,0x00,0x00,0x00,0x04,0xc8,0x67,0x00,0x20,
        0x00,0x49,0x08,0x47,0xcd,0x55,0x00,0x20,0x81,0x6a,0x7d,0x20,0x00,0x02,0x81,0x42,
        0x02,0xd3,0x08,0x1a,0x01,0x22,0x00,0xe0,0x40,0x1a,0x00,0x2a,0x04,0xd0,0x60,0x43,
        0xeb,0xf7,0xa4,0xfc,0x20,0x1a,0x03,0xe0,0x60,0x43,0xeb,0xf7,0xf8,0xb5,0x00,0x24,
        0x1c,0x48,0x00,0x78,0x00,0x28,0x2d,0xd1,0x1b,0x4e,0x70,0x68,0x1b,0x4d,0x00,0x28,
        0x09,0xda,0x1b,0x4f,0x01,0x21,0x0b,0x20,0xb8,0x47,0x01,0x21,0x0b,0x20,0xa8,0x47,
        0x70,0x68,0x00,0x28,0xf6,0xdb,0x00,0x21,0x0b,0x20,0xa8,0x47,0x15,0x4e,0x0a,0x20,
        0xb0,0x47,0x05,0x27,0x3f,0x07,0xf8,0x69,0x05,0x05,0x2d,0x0d,0x00,0x2d,0x04,0xd1,
        0x11,0x48,0xc0,0x68,0x80,0x05,0x80,0x0e,0x0c,0xd0,0x64,0x2c,0x0a,0xd2,0x14,0x20,
        0xb0,0x47,0xf8,0x69,0x00,0x05,0x00,0x0d,0xa8,0x42,0x04,0xd9,0x05,0x46,0x64,0x1c,
        0x64,0x2c,0xf4,0xd3,0xf8,0xbd,0x03,0x49,0x01,0x20,0x08,0x70,0x07,0x49,0x08,0x70,
        0xf8,0xbd,0x00,0x00,0x01,0x10,0x00,0x20,0x80,0x00,0x80,0x45,0x55,0x24,0x00,0x00,
        0x91,0x23,0x00,0x00,0xa1,0x3b,0x00,0x00,0x00,0x00,0xc0,0x52,0x00,0x10,0x00,0x20,
        0x10,0xb5,0x18,0x48,0x01,0x68,0x40,0x29,0x01,0xd2,0x40,0x21,0x01,0x60,0x80,0x7a,
        0xc0,0x07,0x01,0xd0,0x00,0x20,0x10,0xbd,0x13,0x48,0x80,0x47,0x05,0x20,0x00,0x07,
        0xc0,0x69,0x12,0x49,0x00,0x05,0x04,0xd0,0x08,0x78,0x01,0x28,0x14,0xd0,0x02,0x28,
        0x12,0xd0,0x08,0x78,0x00,0x28,0x08,0xd0,0x01,0x28,0x06,0xd0,0x02,0x28,0x04,0xd0,
        0x0b,0x48,0x80,0x47,0x0b,0x49,0x32,0x20,0x88,0x47,0x0b,0x49,0x04,0x20,0x88,0x47,
        0x0a,0x48,0x80,0x47,0x00,0x20,0x10,0xbd,0x09,0x49,0x04,0x20,0x88,0x47,0x01,0x20,
        0x10,0xbd,0x00,0x00,0xb8,0x67,0x00,0x20,0x05,0x93,0x00,0x00,0x18,0x10,0x00,0x20,
        0x25,0x4b,0x01,0x00,0xa1,0x3b,0x00,0x00,0x41,0x44,0x01,0x00,0xaf,0x4a,0x01,0x00,
        0x89,0x44,0x01,0x00,0xf0,0xb5,0x8d,0xb0,0x04,0x46,0x6c,0x49,0x04,0xa8,0x88,0x47,
        0x7c,0x20,0x00,0x5b,0x03,0x90,0x00,0x25,0x00,0x20,0x02,0x90,0x01,0x20,0x80,0xf3,
        0x10,0x88,0x60,0x6c,0x00,0x21,0x81,0xf3,0x10,0x88,0x26,0x46,0x60,0x36,0x00,0x28,
        0x6b,0xd0,0x21,0x46,0x44,0x31,0x0c,0x91,0x28,0x39,0x0b,0x91,0x64,0x31,0x0a,0x91,
        0x81,0x88,0xca,0x00,0x5e,0x49,0x51,0x18,0xc9,0x8c,0xc9,0x0b,0x00,0x29,0x5c,0xd0,
        0x01,0x21,0x81,0xf3,0x10,0x88,0x00,0x68,0x01,0x90,0x5a,0x49,0x0c,0x98,0x88,0x47,
        0x07,0x46,0xe0,0x69,0x00,0x28,0x03,0xd0,0x00,0x20,0x00,0x28,0x02,0xd0,0x08,0xe0,
        0x01,0x20,0xfa,0xe7,0x53,0x49,0x0b,0x98,0x88,0x47,0x00,0x28,0x01,0xd0,0x52,0x49,
        0x88,0x47,0x00,0x20,0x80,0xf3,0x10,0x88,0xb8,0x88,0x4d,0x49,0xc0,0x00,0x40,0x18,
        0xc2,0x8c,0xd2,0x0b,0xd2,0x03,0xc2,0x84,0xb8,0x88,0x07,0x28,0x1e,0xd2,0x30,0x7e,
        0x40,0x1e,0x30,0x76,0x01,0x20,0x80,0xf3,0x10,0x88,0x20,0x6b,0x00,0x28,0x13,0xd0,
        0x00,0x20,0x00,0x28,0x05,0xd0,0x0a,0x98,0xfb,0x21,0x80,0x79,0x08,0x40,0x0a,0x99,
        0x88,0x71,0x41,0x49,0x38,0x46,0x88,0x47,0x00,0x20,0x80,0xf3,0x10,0x88,0x3f,0x4a,
        0x39,0x7b,0x03,0x98,0x90,0x47,0x15,0xe0,0x01,0x20,0xea,0xe7,0x09,0x28,0x0f,0xd9,
        0xc0,0x00,0x40,0x18,0x00,0x8d,0x00,0x0a,0x00,0x90,0x39,0x49,0x38,0x46,0x88,0x47,
        0x00,0x28,0x07,0xd0,0x00,0x98,0x00,0x28,0x04,0xd0,0x6d,0x1c,0xed,0xb2,0x01,0xe0,
        0x6d,0x1c,0xed,0xb2,0x01,0x98,0x00,0x28,0x9a,0xd1,0x03,0x98,0x07,0x28,0x0e,0xd0,
        0x00,0x2d,0x0c,0xd0,0x01,0x20,0x80,0xf3,0x10,0x88,0x30,0x7e,0x40,0x1b,0x30,0x76,
        0x00,0x20,0x80,0xf3,0x10,0x88,0x2b,0x4a,0x29,0x46,0x03,0x98,0x90,0x47,0x01,0x20,
        0x80,0xf3,0x10,0x88,0x28,0x48,0x23,0x4f,0x09,0x90,0xb8,0x47,0x04,0x46,0x00,0x20,
        0x80,0xf3,0x10,0x88,0x20,0x00,0x36,0xd0,0x1f,0x4e,0xe0,0x88,0x03,0x99,0x88,0x42,
        0x12,0xd1,0x07,0x28,0x07,0xd0,0xa1,0x7a,0x00,0x91,0x20,0x4f,0xe3,0x7a,0x22,0x7b,
        0x21,0x89,0xb8,0x47,0x05,0xe0,0x1e,0x4b,0x21,0x79,0x20,0x89,0x2a,0x46,0x98,0x47,
        0x02,0x90,0x20,0x46,0xb0,0x47,0x03,0xe0,0x1a,0x4a,0x21,0x46,0x04,0xa8,0x90,0x47,
        0x01,0x20,0x80,0xf3,0x10,0x88,0x0f,0x49,0x09,0x98,0x88,0x47,0x04,0x46,0x20,0x00,
        0x0c,0xd1,0x04,0x98,0x00,0x28,0x03,0xd0,0x00,0x20,0x00,0x28,0x02,0xd0,0x05,0xe0,
        0x01,0x20,0xfa,0xe7,0x10,0x4a,0x04,0xa9,0x09,0x98,0x90,0x47,0x00,0x20,0x80,0xf3,
        0x10,0x88,0x00,0x2c,0xc9,0xd1,0x02,0x98,0x0d,0xb0,0xf0,0xbd,0xb5,0x38,0x00,0x00,
        0x40,0x44,0x80,0x50,0x45,0x39,0x00,0x00,0xa5,0x93,0x00,0x00,0x09,0xb8,0x00,0x00,
        0x5d,0x56,0x00,0x00,0x05,0xb7,0x00,0x00,0xb8,0x61,0x00,0x20,0x29,0xb7,0x00,0x00,
        0x35,0x22,0x01,0x00,0x67,0x39,0x00,0x00,0x0f,0x39,0x00,0x00,0xf1,0xb5,0x00,0x24,
        0x15,0x4d,0x16,0x4e,0x01,0x20,0x80,0xf3,0x10,0x88,0x00,0x98,0xa8,0x47,0x00,0x21,
        0x81,0xf3,0x10,0x88,0x00,0x28,0x17,0xd0,0x81,0x88,0x0a,0x46,0x0a,0x3a,0x46,0x2a,
        0x14,0xd2,0xc9,0x00,0x89,0x19,0x09,0x8d,0x0f,0x0a,0x01,0x21,0x81,0xf3,0x10,0x88,
        0x0b,0x49,0x88,0x47,0x00,0x28,0x03,0xd0,0x00,0x2f,0x01,0xd0,0x64,0x1c,0xe4,0xb2,
        0x00,0x20,0x80,0xf3,0x10,0x88,0xdd,0xe7,0x20,0x46,0xf8,0xbd,0xc9,0x1f,0x49,0x29,
        0xd8,0xd3,0x04,0x49,0x88,0x47,0xd5,0xe7,0x45,0x39,0x00,0x00,0x40,0x44,0x80,0x50,
        0x5d,0x56,0x00,0x00,0xa5,0x93,0x00,0x00,0xf1,0xb5,0x92,0xb0,0x12,0x98,0x40,0x30,
        0x11,0x90,0x80,0x8f,0x0e,0x90,0x12,0x98,0x08,0x30,0x0d,0x90,0x12,0x98,0x30,0x30,
        0x0c,0x90,0x12,0x98,0x44,0x30,0x0b,0x90,0x00,0x20,0x0a,0x90,0x01,0x25,0x0c,0x98,
        0x04,0x68,0x00,0x2c,0x03,0xd0,0x00,0x20,0x00,0x28,0x02,0xd0,0x93,0xe0,0x01,0x20,
        0xfa,0xe7,0x0e,0x98,0xc1,0x00,0xf9,0x48,0x08,0x18,0x10,0x90,0xc0,0x8c,0xc0,0x0b,
        0x00,0x28,0x6e,0xd0,0x0e,0x98,0xf6,0x49,0x80,0x00,0x0f,0x90,0x08,0x58,0xa0,0x30,
        0x46,0x79,0x01,0x20,0x71,0x07,0x19,0xd5,0x00,0x2c,0x17,0xd0,0xf1,0x4f,0xb0,0x06,
        0x07,0xd5,0x20,0x7b,0xb8,0x47,0x80,0x07,0x01,0xd4,0x00,0x20,0x06,0xe0,0x01,0x20,
        0x04,0xe0,0x20,0x7b,0xb8,0x47,0xc0,0x07,0x03,0xd0,0x01,0x20,0x00,0x28,0x02,0xd0,
        0x04,0xe0,0x00,0x20,0xfa,0xe7,0x24,0x68,0x00,0x2c,0xe8,0xd1,0x00,0x28,0x62,0xd0,
        0xe5,0x4b,0x00,0x22,0x21,0x46,0x0c,0x98,0x98,0x47,0xa6,0x68,0xe3,0x4a,0x09,0xa9,
        0x30,0x46,0x90,0x47,0x00,0x28,0x56,0xd1,0xa0,0x88,0xdc,0x4f,0xc0,0x00,0xc0,0x19,
        0x40,0x8d,0xdf,0x49,0x41,0x18,0x00,0x20,0x08,0xaa,0x12,0x79,0x00,0x2a,0x06,0xdd,
        0x32,0x5c,0x0a,0x54,0x40,0x1c,0x08,0xaa,0x12,0x79,0x90,0x42,0xf8,0xdb,0xd9,0x49,
        0xa0,0x68,0x88,0x47,0xd2,0x49,0x0f,0x98,0x08,0x58,0xa0,0x30,0x40,0x79,0xc0,0x07,
        0x03,0xd0,0x08,0xa9,0x08,0x79,0x00,0x1d,0x09,0x90,0x08,0xa8,0x01,0x79,0x10,0x98,
        0x02,0x8d,0x09,0x02,0xd2,0xb2,0x0a,0x43,0x02,0x85,0x03,0x21,0x10,0x98,0x02,0x8d,
        0x8a,0x43,0x0a,0x43,0x02,0x85,0x11,0x98,0x80,0x8f,0x00,0x21,0xc0,0x00,0xc0,0x19,
        0xc1,0x84,0x0b,0x98,0x00,0x68,0x00,0x28,0x04,0xd0,0x00,0x20,0x00,0x28,0x03,0xd0,
        0x10,0xe0,0x18,0xe0,0x01,0x20,0xf9,0xe7,0x0b,0x98,0xa1,0x88,0x40,0x68,0xca,0x00,
        0xc1,0x49,0x80,0x88,0x51,0x18,0xc0,0x00,0xc0,0x19,0xc2,0x8c,0xd2,0x0b,0xd2,0x03,
        0x0a,0x43,0xc2,0x84,0xbd,0x4a,0x21,0x46,0x0b,0x98,0x90,0x47,0x12,0x98,0x60,0x30,
        0x01,0x7e,0x49,0x1c,0x01,0x76,0x11,0x98,0x80,0x8f,0x07,0x28,0x08,0xd2,0x0e,0x99,
        0x8a,0x00,0xaf,0x49,0x89,0x58,0xa0,0x31,0x49,0x79,0x49,0x07,0x00,0xd5,0x00,0x25,
        0x0d,0x99,0x09,0x68,0x00,0x29,0x1f,0xd0,0x00,0x21,0x2a,0x46,0x8a,0x43,0x77,0xd0,
        0x07,0x28,0x76,0xd2,0xaf,0x49,0xae,0x48,0x88,0x47,0x09,0x90,0x00,0x20,0x08,0x90,
        0x11,0x98,0x80,0x8f,0x81,0x00,0xa2,0x48,0x40,0x58,0xa0,0x30,0x40,0x79,0xc0,0x07,
        0xc0,0x0f,0x07,0x90,0x0d,0x98,0x06,0x68,0x00,0x2e,0x09,0xd0,0x0d,0x99,0x30,0x68,
        0x08,0x60,0x00,0x28,0x02,0xd0,0x03,0xe0,0x01,0x21,0xde,0xe7,0x0d,0x99,0x48,0x60,
        0x00,0x2e,0x7d,0xd0,0x00,0x20,0x06,0x90,0x12,0x98,0x03,0x90,0x06,0xa8,0x04,0x90,
        0x07,0x9f,0x09,0x9b,0x05,0x97,0x08,0x25,0x75,0x5f,0x00,0x20,0x02,0x90,0x01,0x90,
        0x01,0x20,0x00,0x90,0x11,0x98,0x42,0x8e,0x84,0x46,0xd4,0xb2,0x01,0x20,0x29,0x46,
        0x09,0x31,0x89,0xb2,0x05,0x9f,0x00,0x2f,0x01,0xd0,0x09,0x1d,0x89,0xb2,0xc9,0x00,
        0x08,0x31,0x89,0xb2,0x67,0x46,0x3f,0x8e,0x8f,0x42,0x05,0xd2,0x50,0x3f,0x79,0x05,
        0x0c,0x0e,0xa2,0x42,0x00,0xd2,0xd4,0xb2,0xa5,0x42,0x01,0xdd,0x64,0x08,0x64,0x00,
        0x21,0x46,0x6f,0x1a,0x62,0x1c,0x97,0x42,0x05,0xdb,0x09,0x19,0x40,0x1c,0xc0,0xb2,
        0x6f,0x1a,0x97,0x42,0xf9,0xda,0x98,0x42,0x01,0xd9,0x00,0x25,0x34,0xe0,0x04,0x99,
        0x08,0x70,0x81,0x4f,0x81,0x49,0x7e,0x48,0x88,0x47,0x01,0x46,0x00,0x98,0x00,0x28,
        0x12,0xd0,0x00,0x20,0x00,0x90,0xb0,0x7a,0x80,0x07,0x80,0x0f,0x01,0x28,0x09,0xd0,
        0x02,0x20,0x88,0x72,0xf0,0x68,0xc2,0x88,0x02,0x92,0x80,0x88,0x01,0x90,0x09,0xe0,
        0x62,0xe0,0x58,0xe0,0x01,0x20,0xf4,0xe7,0x01,0x20,0x88,0x72,0x02,0x98,0x00,0x19,
        0x80,0xb2,0x02,0x90,0xa5,0x42,0x01,0xdd,0xcc,0x72,0x00,0xe0,0xcd,0x72,0x01,0x98,
        0xc8,0x80,0x02,0x98,0x08,0x81,0x28,0x1b,0x05,0xb2,0x00,0x2d,0x22,0xdc,0x01,0x22,
        0x05,0x9b,0x03,0x98,0xb8,0x47,0x01,0x25,0x00,0x2d,0x20,0xd0,0x68,0x46,0x00,0x7e,
        0x08,0x99,0x09,0x18,0xc9,0xb2,0x08,0x91,0x09,0x99,0x08,0x1a,0x80,0xb2,0x00,0xe0,
        0x35,0xe0,0x09,0x90,0x12,0x98,0xc0,0x69,0x00,0x28,0x1a,0xd0,0x12,0x98,0x00,0x6a,
        0x06,0x60,0x12,0x98,0x06,0x62,0x00,0x20,0x30,0x60,0x08,0x98,0x0a,0x28,0x13,0xd9,
        0x00,0x25,0x24,0xe0,0x00,0x22,0x05,0x9b,0x03,0x98,0xb8,0x47,0xaa,0xe7,0x0d,0x98,
        0x00,0x68,0x00,0x28,0x01,0xd1,0x0d,0x99,0x4e,0x60,0x30,0x60,0x0d,0x98,0x06,0x60,
        0x15,0xe0,0x12,0x98,0xc6,0x61,0xe4,0xe7,0x0d,0x98,0x06,0x68,0x00,0x2e,0x06,0xd0,
        0x0d,0x99,0x30,0x68,0x08,0x60,0x00,0x28,0x01,0xd1,0x0d,0x99,0x48,0x60,0x00,0x2e,
        0x00,0xd0,0x4f,0xe7,0x03,0xe0,0x4a,0x4a,0x0d,0x99,0x0b,0x98,0x90,0x47,0x00,0x2d,
        0x02,0xd0,0x48,0x49,0x0d,0x98,0x88,0x47,0x0b,0x98,0x00,0x68,0x00,0x28,0x03,0xd0,
        0x00,0x20,0x00,0x28,0x02,0xd0,0x4b,0xe0,0x01,0x20,0xfa,0xe7,0x11,0x98,0x80,0x8f,
        0x07,0x28,0x45,0xd2,0x12,0x98,0x80,0x30,0xc0,0x78,0x04,0x28,0x40,0xd1,0x3b,0x49,
        0x37,0x48,0x88,0x47,0x00,0x28,0x3b,0xd0,0x82,0x88,0x00,0x21,0x2b,0x4b,0xd2,0x00,
        0xd2,0x18,0x51,0x85,0x81,0x88,0xc9,0x00,0xc9,0x18,0x8a,0x8d,0x54,0x04,0x64,0x0c,
        0x00,0x22,0x8c,0x85,0x81,0x88,0xc9,0x00,0xc9,0x18,0x0c,0x8d,0xe4,0xb2,0x0c,0x85,
        0x81,0x88,0x01,0x24,0xc9,0x00,0xc9,0x18,0x0d,0x8d,0xad,0x08,0xad,0x00,0x25,0x43,
        0x0d,0x85,0x81,0x88,0xc9,0x00,0xc9,0x18,0xcc,0x8c,0xe4,0x0b,0xe4,0x03,0xcc,0x84,
        0x0b,0x99,0x84,0x88,0x49,0x68,0xe5,0x00,0x1f,0x4c,0x89,0x88,0x2c,0x19,0xc9,0x00,
        0xc9,0x18,0xcb,0x8c,0xdb,0x0b,0xdb,0x03,0x23,0x43,0xcb,0x84,0x0b,0x99,0x09,0x68,
        0x00,0x29,0x0d,0xd0,0x0b,0x99,0x49,0x68,0x08,0x60,0x0b,0x99,0x48,0x60,0x02,0x60,
        0x0b,0x98,0x00,0x68,0x00,0x28,0x06,0xd0,0x00,0x21,0x00,0x29,0x05,0xd0,0x3c,0xe0,
        0x0b,0x99,0x08,0x60,0xf1,0xe7,0x01,0x21,0xf7,0xe7,0x00,0x28,0x0a,0xd0,0x07,0x4a,
        0x81,0x88,0xc9,0x00,0x89,0x18,0xc9,0x8c,0xc9,0x0b,0x00,0x29,0x24,0xd0,0x00,0x68,
        0x00,0x28,0xf5,0xd1,0x00,0x28,0x26,0xd0,0x27,0xe0,0x00,0x00,0x40,0x44,0x80,0x50,
        0x60,0x61,0x00,0x20,0x81,0xaf,0x00,0x00,0x17,0x38,0x00,0x00,0x41,0x03,0x01,0x00,
        0x00,0x40,0x80,0x50,0xa5,0x93,0x00,0x00,0x66,0x04,0x00,0x00,0x67,0x39,0x00,0x00,
        0xe4,0x61,0x00,0x20,0xbb,0x39,0x00,0x00,0x11,0x00,0x01,0x00,0x45,0x39,0x00,0x00,
        0x0f,0x39,0x00,0x00,0xb5,0x38,0x00,0x00,0x81,0x88,0xca,0x00,0x07,0x49,0x51,0x18,
        0x89,0xb2,0x0a,0x91,0xd6,0xe7,0x00,0x20,0x0a,0x90,0x0a,0x99,0x0e,0x98,0x5a,0x22,
        0x50,0x43,0x03,0x4a,0x80,0x18,0x81,0x84,0x13,0xb0,0xf0,0xbd,0x66,0x04,0x00,0x00,
        0x80,0x40,0x80,0x50,0x10,0xb4,0x00,0x23,0x14,0x21,0x02,0x46,0x4a,0x43,0x11,0x49,
        0x7d,0x24,0x09,0x68,0x24,0x02,0xa1,0x42,0x02,0xd9,0x09,0x1b,0x01,0x23,0x00,0xe0,
        0x61,0x1a,0x48,0x43,0x81,0x00,0x41,0x18,0x88,0x0a,0x0c,0x0c,0x00,0x19,0x4c,0x0c,
        0x00,0x19,0x4c,0x0d,0x00,0x19,0x4c,0x0e,0x00,0x19,0xc9,0x0f,0x40,0x18,0xc0,0x08,
        0x00,0x2b,0x01,0xd0,0x10,0x18,0x01,0xe0,0x10,0x1a,0x40,0x1e,0x40,0x1e,0x10,0xbc,
        0x70,0x47,0x00,0x00,0x54,0x60,0x00,0x20,0xf0,0xb4,0x00,0x23,0x18,0x4c,0xe5,0x6b,
        0x18,0x49,0x7d,0x22,0x09,0x68,0x12,0x02,0x91,0x42,0x02,0xd3,0x8a,0x1a,0x01,0x23,
        0x00,0xe0,0x52,0x1a,0x06,0x46,0x56,0x43,0xf2,0x13,0x51,0x43,0x71,0x1a,0x1e,0x26,
        0x4e,0x43,0x4f,0x10,0xf6,0x19,0x8f,0x11,0xf6,0x19,0x49,0x12,0x71,0x18,0xce,0x13,
        0x00,0x2b,0x01,0xd0,0x80,0x1a,0x00,0xe0,0x10,0x18,0x42,0x19,0x91,0x08,0x8d,0x00,
        0x52,0x1b,0x00,0x2b,0x04,0xd0,0x43,0x01,0x18,0x1a,0x40,0x18,0x80,0x1b,0x03,0xe0,
        0x43,0x01,0x18,0x1a,0x40,0x18,0x80,0x19,0xe2,0x63,0xf0,0xbc,0x70,0x47,0x00,0x00,
        0x80,0x67,0x00,0x20,0x54,0x60,0x00,0x20
    }
};

am_hal_ble_patch_t am_ble_buffer_patch =
{
    .ui32Type = 0xCC,
    .ui32Length = 0x0d38,
    .ui32CRC = 0xf515,
    .pui32Data = am_ble_buffer_patch_data.words,
};

//*****************************************************************************
//
// Patch Name:  NVDS  v1.10  for Apollo3 A1
//
// Bi-directional data fix
// Modulation deviation fix
// Extend patch memory
// Transmit speed patch
// Added AGC table and enabled AGC
// Added local feature support setting
// Fix to connection interval calculation issue with MTK chip sets (OPPO R15 fix)
// Set VCO to 250mv
// Modex auto calibration update
// Fix connection interval calculation issue
// Increase RF LDO ref voltage form 1.0v to 1.1v
// Decrease DAC channel delay cycle
// Increase the VCO swing from 250mv to 300mv
// Fix MD schedule issue  (disabled)
// Fix link loss issue
// Reduce duration from TX to TX
// Optimized power consumption (32K clock drift,sleep clock accuracy,,advertising interval (undirect))
// Date:        2019-01-30
//
//*****************************************************************************
am_hal_ble_buffer(0x00be) am_ble_buffer_nvds_data =
{
    .bytes =
    {
        0x4e,0x56,0x44,0x53, //NVDS_MAGIC_NUMBER
        0x01,0x06,0x06,0xef,0xab,0x23,0x88,0x77,0x56, //bluetooth address
        0x02,0x06,0x0a,0x4e,0x5a,0x38,0x38,0x30,0x31,0x56,0x31,0x41,0x00, //device name
        0x03,0x06,0x01,0x00, //system clock frequency, 00=32MHz  01=24MHz  others=16MHz
        0x07,0x06,0x02,0x00,0x00, //32K clock drift, 0x01f4 = 500 ppm
        0x0c,0x06,0x02,50,0x00, //sleep clock accuracy, 0x01f4 = 500 ppm
        0x08,0x06,0x01,0x00, //01 for BQB qualification, 00 for normal usage
        0x09,0x06,0x01,0x02, //clock source selection, 00 = internal RC32KHz, 02= use Apollo3 MCU 32.768KHz
        0x0a,0x06,0x04,0x00,0x00,0x00,0x00, //eb 0x00000000 = auto detect and low frequency clock calibration
        0x0b,0x06,0x01,0x96, //rx_ifs  0x96 = 150us
        0x23,0x06,0x01,0x95, //tx_ifs  0x95 = 149us
        0x0d,0x06,0x02,0xe8,0x03, //duration allowed for XO32M stabilization from external wakeup signal
        0x0e,0x06,0x02,0xe8,0x03, //duration allowed for XO32M stabilization from internal wakeup signal
        0x0f,0x06,0x02,0x2c,0x01, //duration allowed for radio to leave low power mode
        0x10,0x06,0x04,0x00,0xc2,0x01,0x00, //set UART_BAUDRATE
        0x11,0x06,0x01,0x01, //sleep algorithm enabled
        0x12,0x06,0x01,0x01, //external wake-up support
        0x13,0x06,0x02,0xf4,0x01, //duration of sleep and wake-up algorithm
        0x14,0x06,0x02,0x60,0x00, //BLE Company ID
        0x15,0x06,0x01,0x08, //BLE major version
        0x16,0x06,0x01,0x03, //BLE minor version
        0x17,0x06,0x01,0x29, //BLE SW version build
        0x18,0x06,0x02,0xdc,0x05, //advertising interval (undirect)
        0x19,0x06,0x02,0xe2,0x04, //advertising interval (direct)
        0x20,0x06,0x01,0x01, //agc switch
        0x21,0x06,0x01,0x02, //EA programming latency
        0x22,0x06,0x01,0x00, //EA asap latency
        0x24,0x06,0x04,0x42,0x02,0x60,0x09, //radio TRX timing
        0x25,0x06,0x01,0x11, //modem polarity setting
        0x26,0x06,0x01,0x00, //modem sync setting
        0x27,0x06,0x01,0x02, //BLE reset delay
        0x2d,0x06,0x01,0x00, //2 byte mode switch, 01 to enable
        0x28,0x06,0x02,0xf6,0x3f, //initial agc gain setting
        0x29,0x06,0x01,0x0f, //initial Tx output power, 0x0f is +4dBm
        0x35,0x06,0x01,0x08, //maximum Tx ouput power setting
        0x37,0x06,0x01,0x00, //RC32K calibration control, 0xAA to enable
        0x05,0x06,0x02,0x34,0x00, //no use
        0x04,0x06,0x01,0x20, //internal dvdd voltage level control if using 0.9V from MCU side
        0x00,0x00,0x00,0x00  //dummy
    }
};

am_hal_ble_patch_t am_ble_nvds_patch =
{
    .ui32Type = 0xDD,
    .ui32Length = 0x00be,
    .ui32CRC = 0x7e77,
    .pui32Data = am_ble_buffer_nvds_data.words,
};

//*****************************************************************************
//
// End Doxygen group.
//! @}
//
//*****************************************************************************
